﻿<# 

NAME    : Appx-Package Roll-Back
Version : 0.1
Usage   : ./AppPackageRollBack.ps1 -LogFolder "Folder-Path" -AppName "Appx-Name"   
    
    Appx-Package Installer Script accepts parameters,
                   
                   Mandatory Arguments : LogFolder-path, AppName
                                      
#>


[CmdletBinding()]
Param(
    
   [Parameter(Position=1)]
    $LogFolder,

   [Parameter(Position=2)]
    $AppName
)

# Exit Codes for Appx-Installer: Global Variable. 

[int]$Global:ExitCodeSuccess = 0    # Success.
[int]$Global:ExitCodeError   = 1    # Error with error message.
[int]$Global:ExitLogFolderError   = 2    # Error Log-Folder Path not found or invalid path passed.

# Logging for Appx-Installer.

$LogFileName = "\Appx_RollBack.txt"                                 # Log-File Name                                   
$Global:LogFilePath = -join($LogFolder,$LogFileName)                 # Log-File Path: Global Variable.


if ((-not [string]::IsNullOrEmpty($LogFolder)))
{
    If(!(Test-Path -Path $LogFolder))
    {
        Write-Host "LogFolder location is invalid." `n -BackgroundColor Red
        exit $ExitLogFolderError
    }
    else
    {
        try {
                New-Item -ItemType File -Path $LogFilePath -Force
        }
        catch {
                Write-Host "Error Creating Log-File." `n -BackgroundColor Red
                exit $ExitLogFolderError
        }
    }
}
else
{
    Write-Host "Mandatory Argument: Log-Folder path not passed." `n -BackgroundColor Red
    exit $ExitLogFolderError
}


# Function to get time-stamp for Logging.

function Get-TimeStamp {
    
    return "[{0:MM/dd/yy} {0:HH:mm:ss}]" -f (Get-Date)
    
}

# Starting Appx-Logging.

Add-Content -Path $LogFilePath -Value "   ***   Appx-Package Roll-Back Log   ***   "


# Function to Validate Parameters Passed.

function ValidParameter($Parameter,$ParameterName) {
    
    if (([string]::IsNullOrEmpty($Parameter)))
    {
        Write-Host "Manadatory Argument: $ParameterName not passed. " -BackgroundColor Red `n
        Add-Content -Path $LogFilePath -Value "$(Get-TimeStamp)   Manadatory Argument: $ParameterName not passed."
        exit $ExitCodeError
    }

}


<# 
    Function to fetch details of the Appx-Package already installed/provisioned on the system, 
                and then the Appx-Package is De-Provisioned.
#>


function Fetch_Unistall($AppxPackageName) {
    
    $Appx_provisioned_Package = (Get-AppxProvisionedPackage -Online | where DisplayName –eq $AppxPackageName).PackageName
    
    Add-Content -Path $LogFilePath -Value "$(Get-TimeStamp)   PowerShell Command to Fetch the Appx-Provisioned Package Name: (Get-AppxProvisionedPackage -Online | where DisplayName –eq $AppxPackageName).PackageName"
    
    if ((-not [string]::IsNullOrEmpty($Appx_provisioned_Package)))
    {
        
        Remove-AppxProvisionedPackage -Online -PackageName $Appx_provisioned_Package
        
        Add-Content -Path $LogFilePath -Value "$(Get-TimeStamp)   PowerShell Command to De-Provision Appx-Package: Remove-AppxProvisionedPackage -Online -PackageName $Appx_provisioned_Package"

        if ( !$? )
        {

            Write-Host "Error De-Provisioning Appx-Package:" + $Error[0].Exception.Message -BackgroundColor Red `n
            Add-Content -Path $LogFilePath -Value "$(Get-TimeStamp)   Error De-Provisioning Appx-Package: $Error[0].Exception.Message"
            exit $ExitCodeError
    
        }
        else
        {
    
            Write-Host "Appx-Package De-Provisioned Successfully." -BackgroundColor Green `n
            Add-Content -Path $LogFilePath -Value "$(Get-TimeStamp)   Appx-Package De-Provisioned Successfully." 
            
        }

    }

    else
    {

        Write-Host "Appx-Package not Installed/Provisioned to De-provision it." `n
        Add-Content -Path $LogFilePath -Value "$(Get-TimeStamp)   Appx-Package not Installed/Provisioned to De-provision it.." 
    }

    
}


# Main Script Block.


ValidParameter $AppName 'Appx-Package Name'     
   
Fetch_Unistall $AppName                     # Check if Appx-Package is already present, if present Remove-AppxProvisionedPackage.
   
 



# SIG # Begin signature block
# MIIpFQYJKoZIhvcNAQcCoIIpBjCCKQICAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCijQGGuiFSNL24
# DlR8XKLbRYcVX+nKJmaXbfUjXDBSw6CCDh8wggawMIIEmKADAgECAhAIrUCyYNKc
# TJ9ezam9k67ZMA0GCSqGSIb3DQEBDAUAMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQK
# EwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNV
# BAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0yMTA0MjkwMDAwMDBaFw0z
# NjA0MjgyMzU5NTlaMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAw
# ggIKAoICAQDVtC9C0CiteLdd1TlZG7GIQvUzjOs9gZdwxbvEhSYwn6SOaNhc9es0
# JAfhS0/TeEP0F9ce2vnS1WcaUk8OoVf8iJnBkcyBAz5NcCRks43iCH00fUyAVxJr
# Q5qZ8sU7H/Lvy0daE6ZMswEgJfMQ04uy+wjwiuCdCcBlp/qYgEk1hz1RGeiQIXhF
# LqGfLOEYwhrMxe6TSXBCMo/7xuoc82VokaJNTIIRSFJo3hC9FFdd6BgTZcV/sk+F
# LEikVoQ11vkunKoAFdE3/hoGlMJ8yOobMubKwvSnowMOdKWvObarYBLj6Na59zHh
# 3K3kGKDYwSNHR7OhD26jq22YBoMbt2pnLdK9RBqSEIGPsDsJ18ebMlrC/2pgVItJ
# wZPt4bRc4G/rJvmM1bL5OBDm6s6R9b7T+2+TYTRcvJNFKIM2KmYoX7BzzosmJQay
# g9Rc9hUZTO1i4F4z8ujo7AqnsAMrkbI2eb73rQgedaZlzLvjSFDzd5Ea/ttQokbI
# YViY9XwCFjyDKK05huzUtw1T0PhH5nUwjewwk3YUpltLXXRhTT8SkXbev1jLchAp
# QfDVxW0mdmgRQRNYmtwmKwH0iU1Z23jPgUo+QEdfyYFQc4UQIyFZYIpkVMHMIRro
# OBl8ZhzNeDhFMJlP/2NPTLuqDQhTQXxYPUez+rbsjDIJAsxsPAxWEQIDAQABo4IB
# WTCCAVUwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQUaDfg67Y7+F8Rhvv+
# YXsIiGX0TkIwHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4cD08wDgYDVR0P
# AQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHcGCCsGAQUFBwEBBGswaTAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsGAQUFBzAC
# hjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9v
# dEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2VydC5j
# b20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAcBgNVHSAEFTATMAcGBWeBDAED
# MAgGBmeBDAEEATANBgkqhkiG9w0BAQwFAAOCAgEAOiNEPY0Idu6PvDqZ01bgAhql
# +Eg08yy25nRm95RysQDKr2wwJxMSnpBEn0v9nqN8JtU3vDpdSG2V1T9J9Ce7FoFF
# UP2cvbaF4HZ+N3HLIvdaqpDP9ZNq4+sg0dVQeYiaiorBtr2hSBh+3NiAGhEZGM1h
# mYFW9snjdufE5BtfQ/g+lP92OT2e1JnPSt0o618moZVYSNUa/tcnP/2Q0XaG3Ryw
# YFzzDaju4ImhvTnhOE7abrs2nfvlIVNaw8rpavGiPttDuDPITzgUkpn13c5Ubdld
# AhQfQDN8A+KVssIhdXNSy0bYxDQcoqVLjc1vdjcshT8azibpGL6QB7BDf5WIIIJw
# 8MzK7/0pNVwfiThV9zeKiwmhywvpMRr/LhlcOXHhvpynCgbWJme3kuZOX956rEnP
# LqR0kq3bPKSchh/jwVYbKyP/j7XqiHtwa+aguv06P0WmxOgWkVKLQcBIhEuWTatE
# QOON8BUozu3xGFYHKi8QxAwIZDwzj64ojDzLj4gLDb879M4ee47vtevLt/B3E+bn
# KD+sEq6lLyJsQfmCXBVmzGwOysWGw/YmMwwHS6DTBwJqakAwSEs0qFEgu60bhQji
# WQ1tygVQK+pKHJ6l/aCnHwZ05/LWUpD9r4VIIflXO7ScA+2GRfS0YW6/aOImYIbq
# yK+p/pQd52MbOoZWeE4wggdnMIIFT6ADAgECAhAMIw8D+RyJXSpe6W2wChjmMA0G
# CSqGSIb3DQEBCwUAMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwHhcNMjQxMTEyMDAwMDAwWhcNMjUxMTEx
# MjM1OTU5WjCB7zETMBEGCysGAQQBgjc8AgEDEwJVUzEZMBcGCysGAQQBgjc8AgEC
# EwhEZWxhd2FyZTEdMBsGA1UEDwwUUHJpdmF0ZSBPcmdhbml6YXRpb24xEDAOBgNV
# BAUTBzUyODAzOTQxCzAJBgNVBAYTAlVTMQ4wDAYDVQQIEwVUZXhhczETMBEGA1UE
# BxMKUm91bmQgUm9jazEfMB0GA1UEChMWRGVsbCBUZWNobm9sb2dpZXMgSW5jLjEY
# MBYGA1UECxMPQ2xpZW50IFNvZnR3YXJlMR8wHQYDVQQDExZEZWxsIFRlY2hub2xv
# Z2llcyBJbmMuMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEApEoeqEUs
# dZ32U2mG6sAJHxz/GWksveUj0v3F76QWOdmpmt0y12/rPeDTWnM8wqbWN+zrPkvJ
# kNcfNn7MFvXOYNm2hlKk5Hh1+KafmEYtddMo2hLa/OXq2+7j74ZO0NiBcx9t/auh
# cGK7HU2NL0HSnJg2vRcXHIhae4ObuYoTyGtTp+1BhW6nQfStdl9MK77IFG6ysEmR
# J6/HThfW+LiwwKC4Xb4O05AMM3m8o7m4/OqZ41ryiCl/SWrEODs9kQIUEhPV+5gE
# tKKHocPxLGerA5cOh4mp3n3vCR2jg68w2Q6o8BqAwTl3ZBoIAF6QFJf58Q2bC2uD
# cnfBSitF2BVWPknLYWKvf+f40oz/rIZR59L1YJyfXSDqzYVRAZzPSgtLTIyqKXQL
# NgIFuLhKau6oJUTYg5gVD0XLa3GuBAdZ7vawgbhy0557O5UVSKYQM5KSBWLcxFWZ
# KLtCWjy7nCjC/3bxFGIjsAV1L0sUaTp8uNKelfY3Zwo25/7oFCuXzkunAgMBAAGj
# ggICMIIB/jAfBgNVHSMEGDAWgBRoN+Drtjv4XxGG+/5hewiIZfROQjAdBgNVHQ4E
# FgQUj+GRne5IitwePzNKQ37PnBQQk4AwPQYDVR0gBDYwNDAyBgVngQwBAzApMCcG
# CCsGAQUFBwIBFhtodHRwOi8vd3d3LmRpZ2ljZXJ0LmNvbS9DUFMwDgYDVR0PAQH/
# BAQDAgeAMBMGA1UdJQQMMAoGCCsGAQUFBwMDMIG1BgNVHR8Ega0wgaowU6BRoE+G
# TWh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNENvZGVT
# aWduaW5nUlNBNDA5NlNIQTM4NDIwMjFDQTEuY3JsMFOgUaBPhk1odHRwOi8vY3Js
# NC5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRDb2RlU2lnbmluZ1JTQTQw
# OTZTSEEzODQyMDIxQ0ExLmNybDCBlAYIKwYBBQUHAQEEgYcwgYQwJAYIKwYBBQUH
# MAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBcBggrBgEFBQcwAoZQaHR0cDov
# L2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0Q29kZVNpZ25p
# bmdSU0E0MDk2U0hBMzg0MjAyMUNBMS5jcnQwCQYDVR0TBAIwADANBgkqhkiG9w0B
# AQsFAAOCAgEAVae7xPmmUpmMGAue/qi8M2OL3V7Szx7tATAvj6C/BNF16K/uRZCA
# wh2F1EdoEn7zrqqW9mOt00fnet5a+VlgPNx7r7qHcK/3SrVJcBCIkc3SDeqU+Qw7
# 2C4WTv2z5zzs0kiFiZU0Vsdd0paZ/RymSQ9bIbRxDULiGxG2qrCwxgn8vVxm8o6N
# nWkeOeZhCi1b1lgwXQqpKnDdfPOd0toZ5uqO+Iii5ZhRqmnuxXtu0lGrN34Wnm8I
# FM2lVuRwAX+nPmQLvC5up82otQylg1BvT9erNQkLyaqGtDojtM7DQXpphTnqxOW2
# ye7xSvaL7wvhIr+WHqw1ygWyI1h5Ej3QJoFroZ7HmyWjKvAEEY19qL4M+QwpeYo3
# JsjjP03eFsY8QSw56WEzQE3bwz7PT6MuwCrbnrjcmu6lew2sm9kzgGyCbhM22MXm
# eNkMR2/0yv2bFAkMRUWTZTzb0XFpsXwfcndEVI6dLrr/UQTJSmEkTgpIoIDa6LP8
# ibZyQyWKiOrpdv41Ghex7dC3KnXSdUkbZ3NikVLeRMlz2SwtcrMJ3ZZRpS4brjFN
# qbb3miTfyAHEQDTvS0nstqYMwBHhV2dVvmHDRfjlRIG2A4q3wmBYeA/mw51/tdnn
# tluh7+DDE+E/hAs+Mb0N47ruOqXlxhR/qIVggmJdkNqhGHGn7Fhk7EwxghpMMIIa
# SAIBATB9MGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjFB
# MD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcgUlNBNDA5
# NiBTSEEzODQgMjAyMSBDQTECEAwjDwP5HIldKl7pbbAKGOYwDQYJYIZIAWUDBAIB
# BQCggaYwGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEO
# MAwGCisGAQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIDPAwzdKkQtexypDTmGgevYK
# npbREy5nod0zbciwaGV7MDoGCisGAQQBgjcCAQwxLDAqoCiAJgBEAGUAbABsACAA
# VQBwAGQAYQB0AGUAIABQAGEAYwBrAGEAZwBlMA0GCSqGSIb3DQEBAQUABIIBgJGd
# xME0sJMx6EmUG54K/xzwg8551V0Xhi6FUqMrdQ88WdTaBsIG9oHe+Q1D/O/M9i85
# pMlIHRDpLJ5LkvdLPrWr9IOfqa4Mktvl2QFUaFIsrxoFdpa2kWD0jVBnPP0oJfm9
# bQ05OYR/pHg+a8CjSB01XyvuDe10RKTaPUJMYUt+9yVakFgViTLqlkPtankWU6e5
# bVTIhIi8JG9UZzGLgc89MsLzhOS8Sun15lx1PRi+H1ox52QZk+jk7wAPHWuj1kfB
# NoScrJ4Rj5e95H0u+KlV4aZ/6vIvKkTl5kg1EDu8Ew9lmhKQ2QSKBS7tsP2HElgN
# zjmMPntZrL3hyg7NFmKe/kZQ7+gCpkshwzhIjabS8oCuzHopnr7P04m00n7BQB5Y
# GXUHyGa4PslDR52llbuvVuD1uKpGIoEaXix3w2fJS4JWb5aLfEY7Hb5kb8Zj9Y1a
# IvShiIiw4uC9tM3DTql44PDMuZOl9FeKG+udRUm4R7dGS4LuuuYLr1Y8o9GH5KGC
# F3cwghdzBgorBgEEAYI3AwMBMYIXYzCCF18GCSqGSIb3DQEHAqCCF1AwghdMAgED
# MQ8wDQYJYIZIAWUDBAIBBQAweAYLKoZIhvcNAQkQAQSgaQRnMGUCAQEGCWCGSAGG
# /WwHATAxMA0GCWCGSAFlAwQCAQUABCDnGCVhTNZ61BOS3JypbG+1cHYTMct5qRdP
# D/IjTU9lLwIRAMDS+fdHxmW7XsbWh6weSCoYDzIwMjUwODA3MTEwMjU5WqCCEzow
# ggbtMIIE1aADAgECAhAKgO8YS43xBYLRxHanlXRoMA0GCSqGSIb3DQEBCwUAMGkx
# CzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjFBMD8GA1UEAxM4
# RGlnaUNlcnQgVHJ1c3RlZCBHNCBUaW1lU3RhbXBpbmcgUlNBNDA5NiBTSEEyNTYg
# MjAyNSBDQTEwHhcNMjUwNjA0MDAwMDAwWhcNMzYwOTAzMjM1OTU5WjBjMQswCQYD
# VQQGEwJVUzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lD
# ZXJ0IFNIQTI1NiBSU0E0MDk2IFRpbWVzdGFtcCBSZXNwb25kZXIgMjAyNSAxMIIC
# IjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA0EasLRLGntDqrmBWsytXum9R
# /4ZwCgHfyjfMGUIwYzKomd8U1nH7C8Dr0cVMF3BsfAFI54um8+dnxk36+jx0Tb+k
# +87H9WPxNyFPJIDZHhAqlUPt281mHrBbZHqRK71Em3/hCGC5KyyneqiZ7syvFXJ9
# A72wzHpkBaMUNg7MOLxI6E9RaUueHTQKWXymOtRwJXcrcTTPPT2V1D/+cFllESvi
# H8YjoPFvZSjKs3SKO1QNUdFd2adw44wDcKgH+JRJE5Qg0NP3yiSyi5MxgU6cehGH
# r7zou1znOM8odbkqoK+lJ25LCHBSai25CFyD23DZgPfDrJJJK77epTwMP6eKA0kW
# a3osAe8fcpK40uhktzUd/Yk0xUvhDU6lvJukx7jphx40DQt82yepyekl4i0r8OEp
# s/FNO4ahfvAk12hE5FVs9HVVWcO5J4dVmVzix4A77p3awLbr89A90/nWGjXMGn7F
# QhmSlIUDy9Z2hSgctaepZTd0ILIUbWuhKuAeNIeWrzHKYueMJtItnj2Q+aTyLLKL
# M0MheP/9w6CtjuuVHJOVoIJ/DtpJRE7Ce7vMRHoRon4CWIvuiNN1Lk9Y+xZ66laz
# s2kKFSTnnkrT3pXWETTJkhd76CIDBbTRofOsNyEhzZtCGmnQigpFHti58CSmvEyJ
# cAlDVcKacJ+A9/z7eacCAwEAAaOCAZUwggGRMAwGA1UdEwEB/wQCMAAwHQYDVR0O
# BBYEFOQ7/PIx7f391/ORcWMZUEPPYYzoMB8GA1UdIwQYMBaAFO9vU0rp5AZ8esri
# kFb2L9RJ7MtOMA4GA1UdDwEB/wQEAwIHgDAWBgNVHSUBAf8EDDAKBggrBgEFBQcD
# CDCBlQYIKwYBBQUHAQEEgYgwgYUwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRp
# Z2ljZXJ0LmNvbTBdBggrBgEFBQcwAoZRaHR0cDovL2NhY2VydHMuZGlnaWNlcnQu
# Y29tL0RpZ2lDZXJ0VHJ1c3RlZEc0VGltZVN0YW1waW5nUlNBNDA5NlNIQTI1NjIw
# MjVDQTEuY3J0MF8GA1UdHwRYMFYwVKBSoFCGTmh0dHA6Ly9jcmwzLmRpZ2ljZXJ0
# LmNvbS9EaWdpQ2VydFRydXN0ZWRHNFRpbWVTdGFtcGluZ1JTQTQwOTZTSEEyNTYy
# MDI1Q0ExLmNybDAgBgNVHSAEGTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEwDQYJ
# KoZIhvcNAQELBQADggIBAGUqrfEcJwS5rmBB7NEIRJ5jQHIh+OT2Ik/bNYulCrVv
# hREafBYF0RkP2AGr181o2YWPoSHz9iZEN/FPsLSTwVQWo2H62yGBvg7ouCODwrx6
# ULj6hYKqdT8wv2UV+Kbz/3ImZlJ7YXwBD9R0oU62PtgxOao872bOySCILdBghQ/Z
# LcdC8cbUUO75ZSpbh1oipOhcUT8lD8QAGB9lctZTTOJM3pHfKBAEcxQFoHlt2s9s
# XoxFizTeHihsQyfFg5fxUFEp7W42fNBVN4ueLaceRf9Cq9ec1v5iQMWTFQa0xNqI
# tH3CPFTG7aEQJmmrJTV3Qhtfparz+BW60OiMEgV5GWoBy4RVPRwqxv7Mk0Sy4QHs
# 7v9y69NBqycz0BZwhB9WOfOu/CIJnzkQTwtSSpGGhLdjnQ4eBpjtP+XB3pQCtv4E
# 5UCSDag6+iX8MmB10nfldPF9SVD7weCC3yXZi/uuhqdwkgVxuiMFzGVFwYbQsiGn
# oa9F5AaAyBjFBtXVLcKtapnMG3VH3EmAp/jsJ3FVF3+d1SVDTmjFjLbNFZUWMXuZ
# yvgLfgyPehwJVxwC+UpX2MSey2ueIu9THFVkT+um1vshETaWyQo8gmBto/m3acaP
# 9QsuLj3FNwFlTxq25+T4QwX9xa6ILs84ZPvmpovq90K8eWyG2N01c4IhSOxqt81n
# MIIGtDCCBJygAwIBAgIQDcesVwX/IZkuQEMiDDpJhjANBgkqhkiG9w0BAQsFADBi
# MQswCQYDVQQGEwJVUzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3
# d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdpQ2VydCBUcnVzdGVkIFJvb3Qg
# RzQwHhcNMjUwNTA3MDAwMDAwWhcNMzgwMTE0MjM1OTU5WjBpMQswCQYDVQQGEwJV
# UzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4xQTA/BgNVBAMTOERpZ2lDZXJ0IFRy
# dXN0ZWQgRzQgVGltZVN0YW1waW5nIFJTQTQwOTYgU0hBMjU2IDIwMjUgQ0ExMIIC
# IjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAtHgx0wqYQXK+PEbAHKx126NG
# aHS0URedTa2NDZS1mZaDLFTtQ2oRjzUXMmxCqvkbsDpz4aH+qbxeLho8I6jY3xL1
# IusLopuW2qftJYJaDNs1+JH7Z+QdSKWM06qchUP+AbdJgMQB3h2DZ0Mal5kYp77j
# YMVQXSZH++0trj6Ao+xh/AS7sQRuQL37QXbDhAktVJMQbzIBHYJBYgzWIjk8eDrY
# hXDEpKk7RdoX0M980EpLtlrNyHw0Xm+nt5pnYJU3Gmq6bNMI1I7Gb5IBZK4ivbVC
# iZv7PNBYqHEpNVWC2ZQ8BbfnFRQVESYOszFI2Wv82wnJRfN20VRS3hpLgIR4hjzL
# 0hpoYGk81coWJ+KdPvMvaB0WkE/2qHxJ0ucS638ZxqU14lDnki7CcoKCz6eum5A1
# 9WZQHkqUJfdkDjHkccpL6uoG8pbF0LJAQQZxst7VvwDDjAmSFTUms+wV/FbWBqi7
# fTJnjq3hj0XbQcd8hjj/q8d6ylgxCZSKi17yVp2NL+cnT6Toy+rN+nM8M7LnLqCr
# O2JP3oW//1sfuZDKiDEb1AQ8es9Xr/u6bDTnYCTKIsDq1BtmXUqEG1NqzJKS4kOm
# xkYp2WyODi7vQTCBZtVFJfVZ3j7OgWmnhFr4yUozZtqgPrHRVHhGNKlYzyjlroPx
# ul+bgIspzOwbtmsgY1MCAwEAAaOCAV0wggFZMBIGA1UdEwEB/wQIMAYBAf8CAQAw
# HQYDVR0OBBYEFO9vU0rp5AZ8esrikFb2L9RJ7MtOMB8GA1UdIwQYMBaAFOzX44LS
# cV1kTN8uZz/nupiuHA9PMA4GA1UdDwEB/wQEAwIBhjATBgNVHSUEDDAKBggrBgEF
# BQcDCDB3BggrBgEFBQcBAQRrMGkwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRp
# Z2ljZXJ0LmNvbTBBBggrBgEFBQcwAoY1aHR0cDovL2NhY2VydHMuZGlnaWNlcnQu
# Y29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcnQwQwYDVR0fBDwwOjA4oDagNIYy
# aHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5j
# cmwwIAYDVR0gBBkwFzAIBgZngQwBBAIwCwYJYIZIAYb9bAcBMA0GCSqGSIb3DQEB
# CwUAA4ICAQAXzvsWgBz+Bz0RdnEwvb4LyLU0pn/N0IfFiBowf0/Dm1wGc/Do7oVM
# Y2mhXZXjDNJQa8j00DNqhCT3t+s8G0iP5kvN2n7Jd2E4/iEIUBO41P5F448rSYJ5
# 9Ib61eoalhnd6ywFLerycvZTAz40y8S4F3/a+Z1jEMK/DMm/axFSgoR8n6c3nuZB
# 9BfBwAQYK9FHaoq2e26MHvVY9gCDA/JYsq7pGdogP8HRtrYfctSLANEBfHU16r3J
# 05qX3kId+ZOczgj5kjatVB+NdADVZKON/gnZruMvNYY2o1f4MXRJDMdTSlOLh0HC
# n2cQLwQCqjFbqrXuvTPSegOOzr4EWj7PtspIHBldNE2K9i697cvaiIo2p61Ed2p8
# xMJb82Yosn0z4y25xUbI7GIN/TpVfHIqQ6Ku/qjTY6hc3hsXMrS+U0yy+GWqAXam
# 4ToWd2UQ1KYT70kZjE4YtL8Pbzg0c1ugMZyZZd/BdHLiRu7hAWE6bTEm4XYRkA6T
# l4KSFLFk43esaUeqGkH/wyW4N7OigizwJWeukcyIPbAvjSabnf7+Pu0VrFgoiovR
# Diyx3zEdmcif/sYQsfch28bZeUz2rtY/9TCA6TD8dC3JE3rYkrhLULy7Dc90G6e8
# BlqmyIjlgp2+VqsS9/wQD7yFylIz0scmbKvFoW2jNrbM1pD2T7m3XDCCBY0wggR1
# oAMCAQICEA6bGI750C3n79tQ4ghAGFowDQYJKoZIhvcNAQEMBQAwZTELMAkGA1UE
# BhMCVVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2lj
# ZXJ0LmNvbTEkMCIGA1UEAxMbRGlnaUNlcnQgQXNzdXJlZCBJRCBSb290IENBMB4X
# DTIyMDgwMTAwMDAwMFoXDTMxMTEwOTIzNTk1OVowYjELMAkGA1UEBhMCVVMxFTAT
# BgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQd3d3LmRpZ2ljZXJ0LmNvbTEh
# MB8GA1UEAxMYRGlnaUNlcnQgVHJ1c3RlZCBSb290IEc0MIICIjANBgkqhkiG9w0B
# AQEFAAOCAg8AMIICCgKCAgEAv+aQc2jeu+RdSjwwIjBpM+zCpyUuySE98orYWcLh
# Kac9WKt2ms2uexuEDcQwH/MbpDgW61bGl20dq7J58soR0uRf1gU8Ug9SH8aeFaV+
# vp+pVxZZVXKvaJNwwrK6dZlqczKU0RBEEC7fgvMHhOZ0O21x4i0MG+4g1ckgHWMp
# Lc7sXk7Ik/ghYZs06wXGXuxbGrzryc/NrDRAX7F6Zu53yEioZldXn1RYjgwrt0+n
# MNlW7sp7XeOtyU9e5TXnMcvak17cjo+A2raRmECQecN4x7axxLVqGDgDEI3Y1Dek
# LgV9iPWCPhCRcKtVgkEy19sEcypukQF8IUzUvK4bA3VdeGbZOjFEmjNAvwjXWkmk
# wuapoGfdpCe8oU85tRFYF/ckXEaPZPfBaYh2mHY9WV1CdoeJl2l6SPDgohIbZpp0
# yt5LHucOY67m1O+SkjqePdwA5EUlibaaRBkrfsCUtNJhbesz2cXfSwQAzH0clcOP
# 9yGyshG3u3/y1YxwLEFgqrFjGESVGnZifvaAsPvoZKYz0YkH4b235kOkGLimdwHh
# D5QMIR2yVCkliWzlDlJRR3S+Jqy2QXXeeqxfjT/JvNNBERJb5RBQ6zHFynIWIgnf
# fEx1P2PsIV/EIFFrb7GrhotPwtZFX50g/KEexcCPorF+CiaZ9eRpL5gdLfXZqbId
# 5RsCAwEAAaOCATowggE2MA8GA1UdEwEB/wQFMAMBAf8wHQYDVR0OBBYEFOzX44LS
# cV1kTN8uZz/nupiuHA9PMB8GA1UdIwQYMBaAFEXroq/0ksuCMS1Ri6enIZ3zbcgP
# MA4GA1UdDwEB/wQEAwIBhjB5BggrBgEFBQcBAQRtMGswJAYIKwYBBQUHMAGGGGh0
# dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBDBggrBgEFBQcwAoY3aHR0cDovL2NhY2Vy
# dHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9vdENBLmNydDBFBgNV
# HR8EPjA8MDqgOKA2hjRodHRwOi8vY3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRB
# c3N1cmVkSURSb290Q0EuY3JsMBEGA1UdIAQKMAgwBgYEVR0gADANBgkqhkiG9w0B
# AQwFAAOCAQEAcKC/Q1xV5zhfoKN0Gz22Ftf3v1cHvZqsoYcs7IVeqRq7IviHGmlU
# Iu2kiHdtvRoU9BNKei8ttzjv9P+Aufih9/Jy3iS8UgPITtAq3votVs/59PesMHqa
# i7Je1M/RQ0SbQyHrlnKhSLSZy51PpwYDE3cnRNTnf+hZqPC/Lwum6fI0POz3A8eH
# qNJMQBk1RmppVLC4oVaO7KTVPeix3P0c2PR3WlxUjG/voVA9/HYJaISfb8rbII01
# YBwCA8sgsKxYoA5AY8WYIsGyWfVVa88nq2x2zm8jLfR+cWojayL/ErhULSd+2DrZ
# 8LaHlv1b0VysGMNNn3O3AamfV6peKOK5lDGCA3wwggN4AgEBMH0waTELMAkGA1UE
# BhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMuMUEwPwYDVQQDEzhEaWdpQ2Vy
# dCBUcnVzdGVkIEc0IFRpbWVTdGFtcGluZyBSU0E0MDk2IFNIQTI1NiAyMDI1IENB
# MQIQCoDvGEuN8QWC0cR2p5V0aDANBglghkgBZQMEAgEFAKCB0TAaBgkqhkiG9w0B
# CQMxDQYLKoZIhvcNAQkQAQQwHAYJKoZIhvcNAQkFMQ8XDTI1MDgwNzExMDI1OVow
# KwYLKoZIhvcNAQkQAgwxHDAaMBgwFgQU3WIwrIYKLTBr2jixaHlSMAf7QX4wLwYJ
# KoZIhvcNAQkEMSIEIDQ5kU8QZ0Jc08dIfBVoFd4QWcWtEMoqmQF5ME8Q4x0bMDcG
# CyqGSIb3DQEJEAIvMSgwJjAkMCIEIEqgP6Is11yExVyTj4KOZ2ucrsqzP+NtJpqj
# NPFGEQozMA0GCSqGSIb3DQEBAQUABIICAL90R0vFcSQ6H2PS54d5h/6KSIhGZCpT
# Xa+AAQl+Uzwk3Mwhjq/dbElmuIFNEH7g+vK4Sflv5g44GCapUAMRt5mgP+iNxDKq
# 0Co84NjkThS3bph14t7qNdlyNkhA+TdHxWBqNxgY69QHVLhndWjkCEQ/fJmuDYFf
# RuKp5KxB+p16OBmgk6SpN8qY6ktAb9ODBKTVJfhkKrAOhsgF5jNQ2z0J7+HI+6op
# STkuF4E7UtNiBMaQBHp5qSysVhh1DPCerDBs+uT743ieqPayXXSg7Hcptmx2Q1wY
# tiwHxqA7BxJexG76Pd1sPcm46GuhR7J4+halz1bp5kLsND4Hnm4vpRGyg2YojhOC
# wkvGIJ5L/Cy2GOq9sijU7prb7kGzxrZoln4b2BFj+wsKE7XErvGTCpU6oGOS97Xn
# 3+Z6tQxT+y3KWucitoYNRZDAXS4So0iZvSu8TqKnIFcbEGNRlG6sESJduPHSoWsJ
# DH5oRgoVMJZ2i5G+17/T2+QH3Hvw85Btarqr7n9a2KFlYQIcdK6nbjf8WDi2ZTa/
# U0DS3V4N8C56NeN2uet1gyZicIPDU7ThQ/dMy3CBgJlRRVf4UuzGs1bu9Wlgl9Ce
# GWQBNzXgL7Dt25qrYT2uRD5LqjIyq3rW7hzPmU3yjeNY7nsDjNuMKETpV0OLsfLI
# lAEKGqZBxrfn
# SIG # End signature block
