/*******************************************************************************
** INTEL CONFIDENTIAL
**
** Copyright (c) 2022-2025 Intel Corporation
**
** This software and the related documents are Intel copyrighted materials,
** and your use of them is governed by the express license under which they
** were provided to you ("License"). Unless the License provides otherwise,
** you may not use, modify, copy, publish, distribute, disclose or transmit
** this software or the related documents without Intel's prior written
** permission.
**
** The source code contained or described herein and all documents related to
** the source code ("Material") are owned by Intel Corporation or its suppliers
** or licensors.
**
** This software and the related documents are provided as is,
** with no express or implied warranties, other than those that are expressly
** stated in the License.
**
**
**
** Copyright (C) 2022-2025 Intel Corporation
** SPDX-License-Identifier: MIT
**
**
** Permission is hereby granted, free of charge, to any person obtaining a copy
** of this software and associated documentation files (the "Software"), to deal
** in the Software without restriction, including without limitation the rights
** to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
** copies of the Software, and to permit persons to whom the Software is
** furnished to do so, subject to the following conditions:
**
** The above copyright notice and this permission notice (including the next
** paragraph) shall be included in all copies or substantial portions of the
** Software.
**
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
** IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
** FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
** AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
** LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
** OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
** SOFTWARE.
**
**
*******************************************************************************/

/******************************************************************************* 
** Usage of Testclient.exe
** Default command line : testclient.exe
**		1. Will print all the available namespace details using GetNode call
**		2. Single iteration, no polling
** Command line with arguments :testclient.exe --namespace=Platform.Extension --iterations=3 --interval=2000 --QueryNode=Platform --GetSchema=Platform --output-format=<csv>
**		1. Print all the IPF namespace nodes under “Platform.Extensions”
**		2. Iterate for 3 times with a polling interval of 2 seconds.
**      3. Print the node under Platform 
**      4. Print the Data Schema of the IPF namespace nodes under Platform 
**      5. Dumps the output to the .csv file format
** 
*******************************************************************************/
#include <iostream>
#include <Windows.h>
#include <sstream>
#include <fstream>
#include <chrono>
#include "IpfClient.h"
#include "Exceptions.h"
#include "SchemaType.h"
#include "nlohmann/json.hpp"

using namespace std;
using Json = nlohmann::json;
using namespace std::chrono;
// An object that can be passed to RegisterEvent as context.
// A pointer to this object will be passed into the SampleEventCallback

typedef struct TIME_DATE_
{
	std::string Time;
	std::string Date;
}TimeDate;

std::string GetTimeinUTC(std::string& Date)
{
	SYSTEMTIME st;
	std::ostringstream SysTime;
	std::ostringstream SysDate;
	GetSystemTime(&st);
	SysDate << st.wDay << "/"
		<< std::setw(2) << std::setfill('0') << st.wMonth << "/"
		<< std::setw(2) << std::setfill('0') << st.wYear << "";

	Date = SysDate.str();

	SysTime << std::setw(2) << std::setfill('0') << st.wHour << ":"
		<< std::setw(2) << std::setfill('0') << st.wMinute << ":"
		<< std::setw(2) << std::setfill('0') << st.wSecond << ":"
		<< std::setw(3) << std::setfill('0') << st.wMilliseconds << "";

	return std::string(SysTime.str());
}

std::string TimeToTimeStr(time_t mTimeStamp)
{
	tm local_time;
	std::stringstream TimeStr;

	errno_t TimeErr = localtime_s(&local_time, &mTimeStamp);
	if (TimeErr != 0) return "";
	TimeStr << local_time.tm_hour << "H_" << local_time.tm_min << "M_" << local_time.tm_sec << "s";

	return TimeStr.str();
}
std::string TimeToDateStr(time_t mTimeStamp)
{
	tm local_time;
	std::stringstream TimeStr;

	errno_t TimeErr = localtime_s(&local_time, &mTimeStamp);
	if (TimeErr != 0) return "";
	TimeStr << local_time.tm_mday << "-" << (local_time.tm_mon + 1) << "-" << "2021_";// TODO:: year is giving junk value , have to debug

	return TimeStr.str();
}

class ContextObject
{
public:
	void Hello(std::string event_data)
	{
		std::cout << "Hello from ContextObject! Event data is " << event_data << std::endl;
	}
};



// Function to be called in case of provider event.
void CallbackFunction(const char* path, const char* event, void* context)
{
	std::cout << "Received event from " << path << ". Event data: " << event << std::endl;
	if (context) {
		static_cast<ContextObject*>(context)->Hello(event);
	}
}

static void ConsolePrint(const std::string& field, const std::string& value)
{
	std::cout << field << std::endl << value << std::endl << std::endl;
}

#define EMPTY_STR   ""
#define FILE_NAME "IPF_Providers_Output"
#define COMMA_SEPARATOR  ","

typedef struct {
	std::string NameSpacePath;
	int iteration;
	DWORD Interval;
}CmdArgs_t;

// //Load and initialize providersa
//Ipf::ClientApiJson ipf;

high_resolution_clock::time_point start;
time_t starttime = 0;
std::string StrDateTime = "";
int IterationCount = 0;
bool IsCSVFile = false;
bool IsSchema_QueryNode = false;


uint32_t AddNodeToList(string strNodeName, Json& jNode, map<string, string>& KeyValueMap)
{
	uint32_t uReturn = 0;
	try
	{
		for (auto itr : jNode.items())
		{
			if (itr.value().type() == Json::value_t::object)
			{
				AddNodeToList(strNodeName + "." + itr.key(), itr.value(), KeyValueMap);
			}
			else if (itr.value().type() == Json::value_t::string)
			{
				KeyValueMap[strNodeName + "." + itr.key()] = itr.value().get<string>();
			}
			else if (itr.value().type() == Json::value_t::null)
			{
				KeyValueMap[strNodeName + "." + itr.key()] = "null";
			}
			else if ((itr.value().type() == Json::value_t::number_integer) ||
				(itr.value().type() == Json::value_t::number_float) ||
				(itr.value().type() == Json::value_t::number_unsigned))
			{
				KeyValueMap[strNodeName + "." + itr.key()] = to_string(itr.value().get<int>());
			}
			else if (itr.value().type() == Json::value_t::array)
			{
				for (size_t nIndex = 0; nIndex < itr.value().size(); ++nIndex)
				{
					AddNodeToList(strNodeName + "." + itr.key() + "." + to_string(nIndex) , itr.value()[nIndex], KeyValueMap);
				}
			}

		}
	}
	catch (exception& e)
	{
		cout << "Exception occured in" << __FUNCTION__ << e.what();
		uReturn = 1;
	}
	return uReturn;
}

uint32_t ConvertToCSV(Json& jData, string& strHeaderList, string& strValueList)
{
	uint32_t uReturn = 0;
	map<string, string> KeyValueMap;
	try
	{
		KeyValueMap.clear();

		for (auto itr : jData.items())
		{
			uReturn = AddNodeToList(itr.key(), itr.value(), KeyValueMap);
		}

		strHeaderList.clear();
		strValueList.clear();
		for (auto& kv : KeyValueMap)
		{
			strHeaderList += kv.first + ",";
			strValueList += kv.second + ",";

		}
		strHeaderList.pop_back();
		strValueList.pop_back();
	}
	catch (exception& e)
	{
		uReturn = 1;
		cout << "Exception occured in" << __FUNCTION__ << e.what();
	}
	return uReturn;
}

int ParseCommandLineArguments(int argc, char* argv[], CmdArgs_t& args, Ipf::ClientApiJson &ipf)
{
	int ret = 0;

	try
	{
		args.NameSpacePath = EMPTY_STR;
		args.iteration = 1;
		args.Interval = 1000;
		std::string key, value;
		std::size_t start_pos, inner_pos, end_pos;

		for (int i = 1; (i < argc); i++)
		{
			key = argv[i];
			start_pos = key.find("--");
			if (start_pos == std::string::npos)
			{
				std::cout << "Please check the argument passed :" << std::endl;
				ret = -1;
				goto exit;
			}
			key = key.substr(start_pos + 2);
			inner_pos = key.find("=");
			if (inner_pos == std::string::npos)
			{
				std::cout << "Please check the argument passed :" << std::endl;
				ret = -1;
				goto exit;
			}
			value = key.substr(inner_pos + 1);
			end_pos = inner_pos - start_pos;
			key = key.substr(start_pos, end_pos);
			transform(key.begin(), key.end(), key.begin(), ::tolower);
			if (key == "namespace")
			{
				args.NameSpacePath = value;
			}
			else if (key == "iterations")
			{
				if (value == EMPTY_STR)
				{
					args.iteration = 1;
				}
				else {
					args.iteration = std::stoul(value);
				}
				IterationCount = args.iteration;
			}
			else if (key == "interval")
			{
				if (value == EMPTY_STR)
				{
					args.Interval = 1000;
				}
				else
				{
					args.Interval = std::stoul(value);
				}
			}
			else if (key == "querynode")
			{
				IsSchema_QueryNode = true;
				if (value == EMPTY_STR)
				{
					auto jsonNode = ipf.QueryNode("Platform");
					ConsolePrint("ipf.QueryNode(\"\")", jsonNode.dump(4));
				}
				else {
					auto jsonNode = ipf.QueryNode(value);
					ConsolePrint("ipf.QueryNode(\"\")", jsonNode.dump(4));
				}
			}
			else if (key == "getschema")
			{
				IsSchema_QueryNode = true;
				if (value == EMPTY_STR)
				{
					std::cout << "Please give the Valid Node for GetSchema " << std::endl;
					std::cout << "For Eg: --GetSchema=Platform.CPU" << std::endl;
				}
				else {
					auto jsonSchema = ipf.GetSchema(value, Ipf::SchemaType::DATA);
					ConsolePrint("ipf.GetSchema(\"\")", jsonSchema.dump(4));
				}
			}
			else if (key == "output-format")
			{
				transform(value.begin(), value.end(), value.begin(), ::tolower);
				if (value == "csv")
				{
					IsCSVFile = true;
				}
				else
				{
					IsCSVFile = false;
				}

			}
			else
			{
				std::cout << "Please check the argument passed :" << std::endl;
				ret = -1;
				goto exit;
			}
		}

	}
	catch (exception& e)
	{
		ret = 1;
		cout << "Exception occured in" << __FUNCTION__ << e.what();
	}

exit:
	return ret;
}


int main(int argc, char** argv, CmdArgs_t& args)
{
	int ret = 0;
	try 
	{
		//Load and initialize providersa
		Ipf::ClientApiJson ipf;
		int timeval = 0;
		DWORD sleepval ;

		std::string FileName = FILE_NAME;
		std::string FileNameJson = FILE_NAME;
		std::string FileHeader = "Date, Time,";
		std::string Date = "";
		std::string Time = "";
		start = high_resolution_clock::now();
		starttime = time(0);
		StrDateTime = TimeToDateStr(starttime) + TimeToTimeStr(starttime);
		string strHeader, strValue;
		Json jGetNode;
		std::vector<Json> jsonKey;
		std::vector<TimeDate> TimeDateVec;
		std::vector<std::string>::iterator itr;
		bool HeaderCount = false;

		if (argc == 1)
		{
			args.NameSpacePath = EMPTY_STR;
			IsCSVFile = false;
			IsSchema_QueryNode = false;
		}
		else
		{
			if (ParseCommandLineArguments(argc, argv, args, ipf) == -1)
			{
				std::cout << "Failure in ParseCommandLineArguments" << std::endl;
				ret = -1;
				goto exit;
			}
		}

		if (IsCSVFile)
		{
			FileName += "_" + StrDateTime + ".csv";
		}
		else
		{
			FileName += "_" + StrDateTime + ".json";
		}

		if(!IsSchema_QueryNode)
		{ 
			fstream file(FileName, ios::out);

			do 
			{
				TimeDate timedate_st = {};
				timedate_st.Time = GetTimeinUTC(timedate_st.Date);
				jGetNode = ipf.GetNode(args.NameSpacePath);
				jsonKey.push_back(jGetNode);
				TimeDateVec.push_back(timedate_st);
				ConsolePrint("ipf.GetNode(\"\")", jGetNode.dump(4));

				Sleep(args.Interval);
				if (timeval == 1)
				{
					HeaderCount = true;
				}
				timeval++;

			} while (timeval < IterationCount);

			if (!IsCSVFile)
			{
				for (auto it = jsonKey.begin(); it != jsonKey.end(); ++it)
					file << *it << endl;

			}
			else
			{
				HeaderCount = true;
				int index = 0;
				for (auto it = jsonKey.begin(); it != jsonKey.end(); ++it)
				{
					if (ConvertToCSV(*it, strHeader, strValue) == 0)
					{
						if (HeaderCount)
						{
							file << FileHeader + strHeader << endl;
							HeaderCount = false;
						}
						file << TimeDateVec[index].Date + COMMA_SEPARATOR + TimeDateVec[index].Time + COMMA_SEPARATOR + strValue << endl;
					}

					index++;
				}

			}

			file.close();
			std::cout << "\t\n --> OutputFile - " << FileName << std::endl;
			auto stop = high_resolution_clock::now();
			auto durationval = duration_cast<milliseconds>(stop - start);
			std::cout << "\t \n --> Time taken to stop collection- " << std::to_string(durationval.count()) << std::endl;
					
					
		}
	}

	// Catch IPF exceptions
	catch (const Ipf::IpfException& e) {
		std::cerr << "Got IpfException when accessing the namespace: " << e.what() << std::endl;
		ret = -1;
		goto exit;
	}

	catch (std::exception& e) {
		std::cerr << "Got standard exception: " << e.what() << std::endl;
		ret = -1;
		goto exit;
	}

	// Catchall
	catch (...) {
		std::cerr << "Unexpected exception" << std::endl;
		ret = -1;
		goto exit;
	}

exit:
	return ret;
}


