﻿<# 

NAME    : Appx-Package Installer
Version : 0.1
Usage   : ./AppPackageInstaller.ps1 -LogFolder "Folder-Path" -AppName "Appx-Name" -MainAppx "Complete path of Main Appx-Package" -AppxLicense "Complete path of Appx-Package License File" -DependencyAppx "Complete path of Dependency Appx-Package, pass multiple Dependency Appx-Package Separated by comma (,)."   
    
    Appx-Package Installer Script accepts parameters,
                   
                   Mandatory Arguments : LogFolder-path, AppName, Main Appx-Package
                   
                   Optional Argument   : Appx-Package license, Dependency Appx-Packages


#>

[CmdletBinding()]
Param(
    
   [Parameter(Position=1)]
    $LogFolder,

   [Parameter(Position=2)]
    $AppName,

   [Parameter(Position=3)]
    $MainAppx,

   [Parameter(Position=4)]
    $AppxLicense,

   [Parameter(Position=5)]
    $DependencyAppx

)


# Exit Codes for Appx-Installer: Global Variable. 

[int]$Global:ExitCodeSuccess = 0    # Success.
[int]$Global:ExitCodeError   = 1    # Error with error message.
[int]$Global:ExitLogFolderError   = 2    # Error Log-Folder Path not found or invalid path passed.


# Logging for Appx-Installer.

$LogFileName = "\Appx_Installer.txt"                                 # Log-File Name                                   
$Global:LogFilePath = -join($LogFolder,$LogFileName)                 # Log-File Path: Global Variable.


if ((-not [string]::IsNullOrEmpty($LogFolder)))
{
    If(!(Test-Path -Path $LogFolder))
    {
        Write-Host "LogFolder location is invalid." `n -BackgroundColor Red
        exit $ExitLogFolderError
    }
    else
    {
        try {
                New-Item -ItemType File -Path $LogFilePath -Force
        }
        catch {
                Write-Host "Error Creating Log-File." `n -BackgroundColor Red
                exit $ExitLogFolderError
        }
    }
}
else
{
    Write-Host "Mandatory Argument: Log-Folder path not passed." `n -BackgroundColor Red
    exit $ExitLogFolderError
}


# Function to get time-stamp for Logging.

function Get-TimeStamp {
    
    return "[{0:MM/dd/yy} {0:HH:mm:ss}]" -f (Get-Date)
    
}


# Starting Appx-Logging.

Add-Content -Path $LogFilePath -Value "   ***   Appx-Package Installation Log   ***   "


# Function to Validate Parameters Passed.

function ValidParameter($Parameter,$ParameterName) {
    
    if (([string]::IsNullOrEmpty($Parameter)))
    {
        Write-Host "Manadatory Argument: $ParameterName not passed. " -BackgroundColor Red `n
        Add-Content -Path $LogFilePath -Value "$(Get-TimeStamp)   Manadatory Argument: $ParameterName not passed."
        exit $ExitCodeError
    }

}

# Function to Test File for it's Existence. 

function ValidFile($TestFile,$FileName) {
    
    If(!(Test-Path -Path $TestFile))
    {
        Write-Host "$FileName NOT FOUND."  -BackgroundColor Red `n
        Add-Content -Path $LogFilePath -Value "$(Get-TimeStamp)   $FileName NOT FOUND."
        exit $ExitCodeError
    }
}


# Function to Install Appx-Package (Provisionig Appx-Package: By default available for all the users.)

    # 1. Installation of Appx-Package with License-File and Dependency Appx-Packages.

function InstallAppx($Appx,$License,$AppxDependency) {
    
        Add-AppxProvisionedPackage -Online -PackagePath $Appx -DependencyPackagePath $AppxDependency -LicensePath $License
    
        Add-Content -Path $LogFilePath -Value "$(Get-TimeStamp)   PowerShell Command to Install/Provision Appx-Package: Add-AppxProvisionedPackage -Online -PackagePath $Appx -DependencyPackagePath $AppxDependency -LicensePath $License"

        if ( !$? )
        {

            Write-Host "Error Installing Appx-Package:" + $Error[0].Exception.Message -BackgroundColor Red `n
            Add-Content -Path $LogFilePath -Value "$(Get-TimeStamp)   Error Installing Appx-Package: $Error[0].Exception.Message "
            exit $ExitCodeError
    
        }
        else
        {
    
            Write-Host "Appx-Package Installed Successfully." -BackgroundColor Green `n
            Add-Content -Path $LogFilePath -Value "$(Get-TimeStamp)   Appx-Package Installed Successfully."
            exit $ExitCodeSuccess
        }


}
    

    # 2. Installation of Appx-Package with only License-File.

function InstallAppxWithLicense($Appx,$License) {
    
        Add-AppxProvisionedPackage -Online -PackagePath $Appx -LicensePath $License

        Add-Content -Path $LogFilePath -Value "$(Get-TimeStamp)   PowerShell Command to Install/Provision Appx-Package: Add-AppxProvisionedPackage -Online -PackagePath $Appx -LicensePath $License"

        if ( !$? )
        {

            Write-Host "Error Installing Appx-Package:" + $Error[0].Exception.Message -BackgroundColor Red `n
            Add-Content -Path $LogFilePath -Value "$(Get-TimeStamp)   Error Installing Appx-Package: $Error[0].Exception.Message "
            exit $ExitCodeError
    
        }
        else
        {
    
            Write-Host "Appx-Package Installed Successfully." -BackgroundColor Green `n
            Add-Content -Path $LogFilePath -Value "$(Get-TimeStamp)   Appx-Package Installed Successfully."
            exit $ExitCodeSuccess
        }


}


    # 3. Installation of Appx-Package with only Dependency Appx-Packages.

function InstallAppxWithDependencyAppx($Appx,$AppxDependency) {
    
        Add-AppxProvisionedPackage -Online -PackagePath $Appx -DependencyPackagePath $AppxDependency -SkipLicense

        Add-Content -Path $LogFilePath -Value "$(Get-TimeStamp)   PowerShell Command to Install/Provision Appx-Package: Add-AppxProvisionedPackage -Online -PackagePath $Appx -DependencyPackagePath $AppxDependency -SkipLicense"

        if ( !$? )
        {

            Write-Host "Error Installing Appx-Package:" + $Error[0].Exception.Message -BackgroundColor Red `n
            Add-Content -Path $LogFilePath -Value "$(Get-TimeStamp)   Error Installing Appx-Package: $Error[0].Exception.Message "
            exit $ExitCodeError
    
        }
        else
        {
    
            Write-Host "Appx-Package Installed Successfully." -BackgroundColor Green `n
            Add-Content -Path $LogFilePath -Value "$(Get-TimeStamp)   Appx-Package Installed Successfully."
            exit $ExitCodeSuccess
        }


}


    # 4. Installation of Appx-Package only.

function InstallAppxDefault($Appx) {
    
        Add-AppxProvisionedPackage -Online -PackagePath $Appx -SkipLicense

        Add-Content -Path $LogFilePath -Value "$(Get-TimeStamp)   PowerShell Command to Install/Provision Appx-Package: Add-AppxProvisionedPackage -Online -PackagePath $Appx -SkipLicense"

        if ( !$? )
        {

            Write-Host "Error Installing Appx-Package:" + $Error[0].Exception.Message -BackgroundColor Red `n
            Add-Content -Path $LogFilePath -Value "$(Get-TimeStamp)   Error Installing Appx-Package: $Error[0].Exception.Message "
            exit $ExitCodeError
    
        }
        else
        {
    
            Write-Host "Appx-Package Installed Successfully." -BackgroundColor Green `n
            Add-Content -Path $LogFilePath -Value "$(Get-TimeStamp)   Appx-Package Installed Successfully."
            exit $ExitCodeSuccess
        }


}


<# 
    Function to check if the Appx-Package already installed on the system, 
                if installed the Appx-Package is De-Provisioned before Installing the New Package.
#>


function CheckAppx_Unistall($AppxPackageName) {
    
    $Appx_provisioned_Package = (Get-AppxProvisionedPackage -Online | where DisplayName –eq $AppxPackageName).PackageName

    Add-Content -Path $LogFilePath -Value "$(Get-TimeStamp)   PowerShell Command to Fetch the Appx-Provisioned Package Name: (Get-AppxProvisionedPackage -Online | where DisplayName –eq $AppxPackageName).PackageName"

    if ((-not [string]::IsNullOrEmpty($Appx_provisioned_Package)))
    {
        
        Remove-AppxProvisionedPackage -Online -PackageName $Appx_provisioned_Package

        Add-Content -Path $LogFilePath -Value "$(Get-TimeStamp)   PowerShell Command to De-Provision Appx-Package: Remove-AppxProvisionedPackage -Online -PackageName $Appx_provisioned_Package"

        if ( !$? )
        {

            Write-Host "Error De-Provisioning existing Appx-Package:" + $Error[0].Exception.Message -BackgroundColor Red `n
            Add-Content -Path $LogFilePath -Value "$(Get-TimeStamp)   Error De-Provisioning existing Appx-Package: $Error[0].Exception.Message"
            exit $ExitCodeError
    
        }
        else
        {
    
            Write-Host "Existing Appx-Package De-Provisioned Successfully." -BackgroundColor Green `n
            Add-Content -Path $LogFilePath -Value "$(Get-TimeStamp)   Existing Appx-Package De-Provisioned Successfully." 
            
        }

    }

    else
    {

        Write-Host "Appx-Package not Installed/Provisioned, Proceeding with Appx-Package Installation/Provisioning." `n
        Add-Content -Path $LogFilePath -Value "$(Get-TimeStamp)   Appx-Package not Installed/Provisioned, Proceeding with Appx-Package Installation/Provisioning." 
    }

    
}

# Function to split Dependency Appx-Packages path and check if each path is valid or not.

function SplitDependencyAppx($DSplit) {
    
    $DependencyPackageArray = $DSplit.Split(',') 
    
    foreach ($D_Array_Ele in $DependencyPackageArray) 
    { 
     
      ValidFile $D_Array_Ele 'Dependency Appx-Package'

    }

}


# Main Script Block.


if( ( -not [string]::IsNullOrEmpty( $DependencyAppx ) ) -and ( -not [string]::IsNullOrEmpty( $AppxLicense ) ) )   # Check if License and Dependency Appx-Packages are passed.
{

   # Validate if all mandatory parameters are passed.
   
   ValidParameter $AppName 'Appx-Package Name'     
   ValidParameter $MainAppx 'Main Appx-Package'
   
   # Validate if the file is present.

   ValidFile $MainAppx 'Main Appx-Package'
   ValidFile $AppxLicense 'Appx License'

   SplitDependencyAppx $DependencyAppx  # Split the Dependency Appx-Packages passed and check if all the file is present.

   CheckAppx_Unistall $AppName          # Check if Appx-Package is already present, if present Remove-AppxProvisionedPackage.

   InstallAppx $MainAppx $AppxLicense $DependencyAppx  # Install or Add-AppxProvisionedPackage.

}
elseif ( ( [string]::IsNullOrEmpty( $DependencyAppx ) ) -and ( -not [string]::IsNullOrEmpty( $AppxLicense ) ) )
{
    
   # Validate if all mandatory parameters are passed.
   
   ValidParameter $AppName 'Appx-Package Name'     
   ValidParameter $MainAppx 'Main Appx-Package'
   
   # Validate if the file is present.

   ValidFile $MainAppx 'Main Appx-Package'
   ValidFile $AppxLicense 'Appx License'

   CheckAppx_Unistall $AppName                     # Check if Appx-Package is already present, if present Remove-AppxProvisionedPackage.
   
   InstallAppxWithLicense $MainAppx $AppxLicense   # Install or Add-AppxProvisionedPackage. 

}
elseif ( ( -not [string]::IsNullOrEmpty( $DependencyAppx ) ) -and ( [string]::IsNullOrEmpty( $AppxLicense ) ) )
{

   # Validate if all mandatory parameters are passed.
   
   ValidParameter $AppName 'Appx-Package Name'     
   ValidParameter $MainAppx 'Main Appx-Package'
   
   # Validate if the file is present.

   ValidFile $MainAppx 'Main Appx-Package'
   
   SplitDependencyAppx $DependencyAppx  # Split the Dependency Appx-Packages passed and check if all the file is present.

   CheckAppx_Unistall $AppName          # Check if Appx-Package is already present, if present Remove-AppxProvisionedPackage.

   InstallAppxWithDependencyAppx $MainAppx $DependencyAppx  # Install or Add-AppxProvisionedPackage.    

}
else
{

   # Validate if all the parameters are passed.
   
   ValidParameter $AppName 'Appx-Package Name'
   ValidParameter $MainAppx 'Main Appx-Package'

   # Validate if the file is present.

   ValidFile $MainAppx 'Main Appx-Package'
   
   CheckAppx_Unistall $AppName          # Check if Appx-Package is already present, if present Remove-AppxProvisionedPackage.

   InstallAppxDefault $MainAppx         # Install or Add-AppxProvisionedPackage.       

}





# SIG # Begin signature block
# MIIo2AYJKoZIhvcNAQcCoIIoyTCCKMUCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCATSWyz3vrbVqGf
# uU5ckk3vMg0/ScOqwOGBD2nwnhOtjqCCDh8wggawMIIEmKADAgECAhAIrUCyYNKc
# TJ9ezam9k67ZMA0GCSqGSIb3DQEBDAUAMGIxCzAJBgNVBAYTAlVTMRUwEwYDVQQK
# EwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3dy5kaWdpY2VydC5jb20xITAfBgNV
# BAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBHNDAeFw0yMTA0MjkwMDAwMDBaFw0z
# NjA0MjgyMzU5NTlaMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAw
# ggIKAoICAQDVtC9C0CiteLdd1TlZG7GIQvUzjOs9gZdwxbvEhSYwn6SOaNhc9es0
# JAfhS0/TeEP0F9ce2vnS1WcaUk8OoVf8iJnBkcyBAz5NcCRks43iCH00fUyAVxJr
# Q5qZ8sU7H/Lvy0daE6ZMswEgJfMQ04uy+wjwiuCdCcBlp/qYgEk1hz1RGeiQIXhF
# LqGfLOEYwhrMxe6TSXBCMo/7xuoc82VokaJNTIIRSFJo3hC9FFdd6BgTZcV/sk+F
# LEikVoQ11vkunKoAFdE3/hoGlMJ8yOobMubKwvSnowMOdKWvObarYBLj6Na59zHh
# 3K3kGKDYwSNHR7OhD26jq22YBoMbt2pnLdK9RBqSEIGPsDsJ18ebMlrC/2pgVItJ
# wZPt4bRc4G/rJvmM1bL5OBDm6s6R9b7T+2+TYTRcvJNFKIM2KmYoX7BzzosmJQay
# g9Rc9hUZTO1i4F4z8ujo7AqnsAMrkbI2eb73rQgedaZlzLvjSFDzd5Ea/ttQokbI
# YViY9XwCFjyDKK05huzUtw1T0PhH5nUwjewwk3YUpltLXXRhTT8SkXbev1jLchAp
# QfDVxW0mdmgRQRNYmtwmKwH0iU1Z23jPgUo+QEdfyYFQc4UQIyFZYIpkVMHMIRro
# OBl8ZhzNeDhFMJlP/2NPTLuqDQhTQXxYPUez+rbsjDIJAsxsPAxWEQIDAQABo4IB
# WTCCAVUwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4EFgQUaDfg67Y7+F8Rhvv+
# YXsIiGX0TkIwHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5nP+e6mK4cD08wDgYDVR0P
# AQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMDMHcGCCsGAQUFBwEBBGswaTAk
# BggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQuY29tMEEGCCsGAQUFBzAC
# hjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9v
# dEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8vY3JsMy5kaWdpY2VydC5j
# b20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAcBgNVHSAEFTATMAcGBWeBDAED
# MAgGBmeBDAEEATANBgkqhkiG9w0BAQwFAAOCAgEAOiNEPY0Idu6PvDqZ01bgAhql
# +Eg08yy25nRm95RysQDKr2wwJxMSnpBEn0v9nqN8JtU3vDpdSG2V1T9J9Ce7FoFF
# UP2cvbaF4HZ+N3HLIvdaqpDP9ZNq4+sg0dVQeYiaiorBtr2hSBh+3NiAGhEZGM1h
# mYFW9snjdufE5BtfQ/g+lP92OT2e1JnPSt0o618moZVYSNUa/tcnP/2Q0XaG3Ryw
# YFzzDaju4ImhvTnhOE7abrs2nfvlIVNaw8rpavGiPttDuDPITzgUkpn13c5Ubdld
# AhQfQDN8A+KVssIhdXNSy0bYxDQcoqVLjc1vdjcshT8azibpGL6QB7BDf5WIIIJw
# 8MzK7/0pNVwfiThV9zeKiwmhywvpMRr/LhlcOXHhvpynCgbWJme3kuZOX956rEnP
# LqR0kq3bPKSchh/jwVYbKyP/j7XqiHtwa+aguv06P0WmxOgWkVKLQcBIhEuWTatE
# QOON8BUozu3xGFYHKi8QxAwIZDwzj64ojDzLj4gLDb879M4ee47vtevLt/B3E+bn
# KD+sEq6lLyJsQfmCXBVmzGwOysWGw/YmMwwHS6DTBwJqakAwSEs0qFEgu60bhQji
# WQ1tygVQK+pKHJ6l/aCnHwZ05/LWUpD9r4VIIflXO7ScA+2GRfS0YW6/aOImYIbq
# yK+p/pQd52MbOoZWeE4wggdnMIIFT6ADAgECAhAMIw8D+RyJXSpe6W2wChjmMA0G
# CSqGSIb3DQEBCwUAMGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwg
# SW5jLjFBMD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcg
# UlNBNDA5NiBTSEEzODQgMjAyMSBDQTEwHhcNMjQxMTEyMDAwMDAwWhcNMjUxMTEx
# MjM1OTU5WjCB7zETMBEGCysGAQQBgjc8AgEDEwJVUzEZMBcGCysGAQQBgjc8AgEC
# EwhEZWxhd2FyZTEdMBsGA1UEDwwUUHJpdmF0ZSBPcmdhbml6YXRpb24xEDAOBgNV
# BAUTBzUyODAzOTQxCzAJBgNVBAYTAlVTMQ4wDAYDVQQIEwVUZXhhczETMBEGA1UE
# BxMKUm91bmQgUm9jazEfMB0GA1UEChMWRGVsbCBUZWNobm9sb2dpZXMgSW5jLjEY
# MBYGA1UECxMPQ2xpZW50IFNvZnR3YXJlMR8wHQYDVQQDExZEZWxsIFRlY2hub2xv
# Z2llcyBJbmMuMIIBojANBgkqhkiG9w0BAQEFAAOCAY8AMIIBigKCAYEApEoeqEUs
# dZ32U2mG6sAJHxz/GWksveUj0v3F76QWOdmpmt0y12/rPeDTWnM8wqbWN+zrPkvJ
# kNcfNn7MFvXOYNm2hlKk5Hh1+KafmEYtddMo2hLa/OXq2+7j74ZO0NiBcx9t/auh
# cGK7HU2NL0HSnJg2vRcXHIhae4ObuYoTyGtTp+1BhW6nQfStdl9MK77IFG6ysEmR
# J6/HThfW+LiwwKC4Xb4O05AMM3m8o7m4/OqZ41ryiCl/SWrEODs9kQIUEhPV+5gE
# tKKHocPxLGerA5cOh4mp3n3vCR2jg68w2Q6o8BqAwTl3ZBoIAF6QFJf58Q2bC2uD
# cnfBSitF2BVWPknLYWKvf+f40oz/rIZR59L1YJyfXSDqzYVRAZzPSgtLTIyqKXQL
# NgIFuLhKau6oJUTYg5gVD0XLa3GuBAdZ7vawgbhy0557O5UVSKYQM5KSBWLcxFWZ
# KLtCWjy7nCjC/3bxFGIjsAV1L0sUaTp8uNKelfY3Zwo25/7oFCuXzkunAgMBAAGj
# ggICMIIB/jAfBgNVHSMEGDAWgBRoN+Drtjv4XxGG+/5hewiIZfROQjAdBgNVHQ4E
# FgQUj+GRne5IitwePzNKQ37PnBQQk4AwPQYDVR0gBDYwNDAyBgVngQwBAzApMCcG
# CCsGAQUFBwIBFhtodHRwOi8vd3d3LmRpZ2ljZXJ0LmNvbS9DUFMwDgYDVR0PAQH/
# BAQDAgeAMBMGA1UdJQQMMAoGCCsGAQUFBwMDMIG1BgNVHR8Ega0wgaowU6BRoE+G
# TWh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0ZWRHNENvZGVT
# aWduaW5nUlNBNDA5NlNIQTM4NDIwMjFDQTEuY3JsMFOgUaBPhk1odHRwOi8vY3Js
# NC5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkRzRDb2RlU2lnbmluZ1JTQTQw
# OTZTSEEzODQyMDIxQ0ExLmNybDCBlAYIKwYBBQUHAQEEgYcwgYQwJAYIKwYBBQUH
# MAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBcBggrBgEFBQcwAoZQaHR0cDov
# L2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0Q29kZVNpZ25p
# bmdSU0E0MDk2U0hBMzg0MjAyMUNBMS5jcnQwCQYDVR0TBAIwADANBgkqhkiG9w0B
# AQsFAAOCAgEAVae7xPmmUpmMGAue/qi8M2OL3V7Szx7tATAvj6C/BNF16K/uRZCA
# wh2F1EdoEn7zrqqW9mOt00fnet5a+VlgPNx7r7qHcK/3SrVJcBCIkc3SDeqU+Qw7
# 2C4WTv2z5zzs0kiFiZU0Vsdd0paZ/RymSQ9bIbRxDULiGxG2qrCwxgn8vVxm8o6N
# nWkeOeZhCi1b1lgwXQqpKnDdfPOd0toZ5uqO+Iii5ZhRqmnuxXtu0lGrN34Wnm8I
# FM2lVuRwAX+nPmQLvC5up82otQylg1BvT9erNQkLyaqGtDojtM7DQXpphTnqxOW2
# ye7xSvaL7wvhIr+WHqw1ygWyI1h5Ej3QJoFroZ7HmyWjKvAEEY19qL4M+QwpeYo3
# JsjjP03eFsY8QSw56WEzQE3bwz7PT6MuwCrbnrjcmu6lew2sm9kzgGyCbhM22MXm
# eNkMR2/0yv2bFAkMRUWTZTzb0XFpsXwfcndEVI6dLrr/UQTJSmEkTgpIoIDa6LP8
# ibZyQyWKiOrpdv41Ghex7dC3KnXSdUkbZ3NikVLeRMlz2SwtcrMJ3ZZRpS4brjFN
# qbb3miTfyAHEQDTvS0nstqYMwBHhV2dVvmHDRfjlRIG2A4q3wmBYeA/mw51/tdnn
# tluh7+DDE+E/hAs+Mb0N47ruOqXlxhR/qIVggmJdkNqhGHGn7Fhk7EwxghoPMIIa
# CwIBATB9MGkxCzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjFB
# MD8GA1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNpZ25pbmcgUlNBNDA5
# NiBTSEEzODQgMjAyMSBDQTECEAwjDwP5HIldKl7pbbAKGOYwDQYJYIZIAWUDBAIB
# BQCggaYwGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGCNwIBCzEO
# MAwGCisGAQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIOeHwIkH1Q0S2BX/+BsIgJgC
# JOT0D08uCWlQUeTYRksCMDoGCisGAQQBgjcCAQwxLDAqoCiAJgBEAGUAbABsACAA
# VQBwAGQAYQB0AGUAIABQAGEAYwBrAGEAZwBlMA0GCSqGSIb3DQEBAQUABIIBgGMu
# 17lDMyToeYA1cDO2uHxEbwOFo1OpAZvnS0ffa8134dsr6NlNctdYvDiEbTD0u3+X
# hRLkSQNZ7ACoSEZ42mdDjUzBl51CX7q8Hp1ZLk2hjVS2fH+mUvimsrZN4mBW4mdN
# AeI5Pn06BYs99qXY/iNhENYwnDQjtYviwLtALK8fEeKuOfNozCCDmHfsMt4k/yti
# R1JDIO+z01WWa/lgCbSodBldFYaDQ5wA9TEX2FuYAw8cjuWmaMy3svpYNF73sd8O
# 3coWwGKJweMQcmHL8wQdtUoU26mgg0kgH7i6hKySCxesTk45VCPqgHWEVAEwyb5C
# g8HjJS5pkLoLYHaUw1muRndJVEUOZ2PC6/DgBWIp0iGOJLZa09vbXW42g0dRMW3l
# mp+1Fij/RHw4/cr7SFwI4CDf0wCFGE2Y6gk4NXYhC3fsiUSqI7Ank3yIXgWV7XBL
# MgaQFyJQEcyFLPs6ndJRT07ISnXwrcmLI5qljSsd5JUj5nc++9i7+zo9JYMNh6GC
# Fzowghc2BgorBgEEAYI3AwMBMYIXJjCCFyIGCSqGSIb3DQEHAqCCFxMwghcPAgED
# MQ8wDQYJYIZIAWUDBAIBBQAweAYLKoZIhvcNAQkQAQSgaQRnMGUCAQEGCWCGSAGG
# /WwHATAxMA0GCWCGSAFlAwQCAQUABCAJ8MNlX1sM8ntGq18VuvYyab36cILCZy3r
# dyHXifVI7AIRAMnHiezILs63UXBbTPH+NE8YDzIwMjUwMjExMTQyMDE5WqCCEwMw
# gga8MIIEpKADAgECAhALrma8Wrp/lYfG+ekE4zMEMA0GCSqGSIb3DQEBCwUAMGMx
# CzAJBgNVBAYTAlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMy
# RGlnaUNlcnQgVHJ1c3RlZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcg
# Q0EwHhcNMjQwOTI2MDAwMDAwWhcNMzUxMTI1MjM1OTU5WjBCMQswCQYDVQQGEwJV
# UzERMA8GA1UEChMIRGlnaUNlcnQxIDAeBgNVBAMTF0RpZ2lDZXJ0IFRpbWVzdGFt
# cCAyMDI0MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAvmpzn/aVIauW
# MLpbbeZZo7Xo/ZEfGMSIO2qZ46XB/QowIEMSvgjEdEZ3v4vrrTHleW1JWGErrjOL
# 0J4L0HqVR1czSzvUQ5xF7z4IQmn7dHY7yijvoQ7ujm0u6yXF2v1CrzZopykD07/9
# fpAT4BxpT9vJoJqAsP8YuhRvflJ9YeHjes4fduksTHulntq9WelRWY++TFPxzZrb
# ILRYynyEy7rS1lHQKFpXvo2GePfsMRhNf1F41nyEg5h7iOXv+vjX0K8RhUisfqw3
# TTLHj1uhS66YX2LZPxS4oaf33rp9HlfqSBePejlYeEdU740GKQM7SaVSH3TbBL8R
# 6HwX9QVpGnXPlKdE4fBIn5BBFnV+KwPxRNUNK6lYk2y1WSKour4hJN0SMkoaNV8h
# yyADiX1xuTxKaXN12HgR+8WulU2d6zhzXomJ2PleI9V2yfmfXSPGYanGgxzqI+Sh
# oOGLomMd3mJt92nm7Mheng/TBeSA2z4I78JpwGpTRHiT7yHqBiV2ngUIyCtd0pZ8
# zg3S7bk4QC4RrcnKJ3FbjyPAGogmoiZ33c1HG93Vp6lJ415ERcC7bFQMRbxqrMVA
# Niav1k425zYyFMyLNyE1QulQSgDpW9rtvVcIH7WvG9sqYup9j8z9J1XqbBZPJ5XL
# ln8mS8wWmdDLnBHXgYly/p1DhoQo5fkCAwEAAaOCAYswggGHMA4GA1UdDwEB/wQE
# AwIHgDAMBgNVHRMBAf8EAjAAMBYGA1UdJQEB/wQMMAoGCCsGAQUFBwMIMCAGA1Ud
# IAQZMBcwCAYGZ4EMAQQCMAsGCWCGSAGG/WwHATAfBgNVHSMEGDAWgBS6FtltTYUv
# cyl2mi91jGogj57IbzAdBgNVHQ4EFgQUn1csA3cOKBWQZqVjXu5Pkh92oFswWgYD
# VR0fBFMwUTBPoE2gS4ZJaHR0cDovL2NybDMuZGlnaWNlcnQuY29tL0RpZ2lDZXJ0
# VHJ1c3RlZEc0UlNBNDA5NlNIQTI1NlRpbWVTdGFtcGluZ0NBLmNybDCBkAYIKwYB
# BQUHAQEEgYMwgYAwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNv
# bTBYBggrBgEFBQcwAoZMaHR0cDovL2NhY2VydHMuZGlnaWNlcnQuY29tL0RpZ2lD
# ZXJ0VHJ1c3RlZEc0UlNBNDA5NlNIQTI1NlRpbWVTdGFtcGluZ0NBLmNydDANBgkq
# hkiG9w0BAQsFAAOCAgEAPa0eH3aZW+M4hBJH2UOR9hHbm04IHdEoT8/T3HuBSyZe
# q3jSi5GXeWP7xCKhVireKCnCs+8GZl2uVYFvQe+pPTScVJeCZSsMo1JCoZN2mMew
# /L4tpqVNbSpWO9QGFwfMEy60HofN6V51sMLMXNTLfhVqs+e8haupWiArSozyAmGH
# /6oMQAh078qRh6wvJNU6gnh5OruCP1QUAvVSu4kqVOcJVozZR5RRb/zPd++PGE3q
# F1P3xWvYViUJLsxtvge/mzA75oBfFZSbdakHJe2BVDGIGVNVjOp8sNt70+kEoMF+
# T6tptMUNlehSR7vM+C13v9+9ZOUKzfRUAYSyyEmYtsnpltD/GWX8eM70ls1V6QG/
# ZOB6b6Yum1HvIiulqJ1Elesj5TMHq8CWT/xrW7twipXTJ5/i5pkU5E16RSBAdOp1
# 2aw8IQhhA/vEbFkEiF2abhuFixUDobZaA0VhqAsMHOmaT3XThZDNi5U2zHKhUs5u
# HHdG6BoQau75KiNbh0c+hatSF+02kULkftARjsyEpHKsF7u5zKRbt5oK5YGwFvgc
# 4pEVUNytmB3BpIiowOIIuDgP5M9WArHYSAR16gc0dP2XdkMEP5eBsX7bf/MGN4K3
# HP50v/01ZHo/Z5lGLvNwQ7XHBx1yomzLP8lx4Q1zZKDyHcp4VQJLu2kWTsKsOqQw
# ggauMIIElqADAgECAhAHNje3JFR82Ees/ShmKl5bMA0GCSqGSIb3DQEBCwUAMGIx
# CzAJBgNVBAYTAlVTMRUwEwYDVQQKEwxEaWdpQ2VydCBJbmMxGTAXBgNVBAsTEHd3
# dy5kaWdpY2VydC5jb20xITAfBgNVBAMTGERpZ2lDZXJ0IFRydXN0ZWQgUm9vdCBH
# NDAeFw0yMjAzMjMwMDAwMDBaFw0zNzAzMjIyMzU5NTlaMGMxCzAJBgNVBAYTAlVT
# MRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjE7MDkGA1UEAxMyRGlnaUNlcnQgVHJ1
# c3RlZCBHNCBSU0E0MDk2IFNIQTI1NiBUaW1lU3RhbXBpbmcgQ0EwggIiMA0GCSqG
# SIb3DQEBAQUAA4ICDwAwggIKAoICAQDGhjUGSbPBPXJJUVXHJQPE8pE3qZdRodbS
# g9GeTKJtoLDMg/la9hGhRBVCX6SI82j6ffOciQt/nR+eDzMfUBMLJnOWbfhXqAJ9
# /UO0hNoR8XOxs+4rgISKIhjf69o9xBd/qxkrPkLcZ47qUT3w1lbU5ygt69OxtXXn
# HwZljZQp09nsad/ZkIdGAHvbREGJ3HxqV3rwN3mfXazL6IRktFLydkf3YYMZ3V+0
# VAshaG43IbtArF+y3kp9zvU5EmfvDqVjbOSmxR3NNg1c1eYbqMFkdECnwHLFuk4f
# sbVYTXn+149zk6wsOeKlSNbwsDETqVcplicu9Yemj052FVUmcJgmf6AaRyBD40Nj
# gHt1biclkJg6OBGz9vae5jtb7IHeIhTZgirHkr+g3uM+onP65x9abJTyUpURK1h0
# QCirc0PO30qhHGs4xSnzyqqWc0Jon7ZGs506o9UD4L/wojzKQtwYSH8UNM/STKvv
# mz3+DrhkKvp1KCRB7UK/BZxmSVJQ9FHzNklNiyDSLFc1eSuo80VgvCONWPfcYd6T
# /jnA+bIwpUzX6ZhKWD7TA4j+s4/TXkt2ElGTyYwMO1uKIqjBJgj5FBASA31fI7tk
# 42PgpuE+9sJ0sj8eCXbsq11GdeJgo1gJASgADoRU7s7pXcheMBK9Rp6103a50g5r
# mQzSM7TNsQIDAQABo4IBXTCCAVkwEgYDVR0TAQH/BAgwBgEB/wIBADAdBgNVHQ4E
# FgQUuhbZbU2FL3MpdpovdYxqII+eyG8wHwYDVR0jBBgwFoAU7NfjgtJxXWRM3y5n
# P+e6mK4cD08wDgYDVR0PAQH/BAQDAgGGMBMGA1UdJQQMMAoGCCsGAQUFBwMIMHcG
# CCsGAQUFBwEBBGswaTAkBggrBgEFBQcwAYYYaHR0cDovL29jc3AuZGlnaWNlcnQu
# Y29tMEEGCCsGAQUFBzAChjVodHRwOi8vY2FjZXJ0cy5kaWdpY2VydC5jb20vRGln
# aUNlcnRUcnVzdGVkUm9vdEc0LmNydDBDBgNVHR8EPDA6MDigNqA0hjJodHRwOi8v
# Y3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVzdGVkUm9vdEc0LmNybDAgBgNV
# HSAEGTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEwDQYJKoZIhvcNAQELBQADggIB
# AH1ZjsCTtm+YqUQiAX5m1tghQuGwGC4QTRPPMFPOvxj7x1Bd4ksp+3CKDaopafxp
# wc8dB+k+YMjYC+VcW9dth/qEICU0MWfNthKWb8RQTGIdDAiCqBa9qVbPFXONASIl
# zpVpP0d3+3J0FNf/q0+KLHqrhc1DX+1gtqpPkWaeLJ7giqzl/Yy8ZCaHbJK9nXzQ
# cAp876i8dU+6WvepELJd6f8oVInw1YpxdmXazPByoyP6wCeCRK6ZJxurJB4mwbfe
# Kuv2nrF5mYGjVoarCkXJ38SNoOeY+/umnXKvxMfBwWpx2cYTgAnEtp/Nh4cku0+j
# Sbl3ZpHxcpzpSwJSpzd+k1OsOx0ISQ+UzTl63f8lY5knLD0/a6fxZsNBzU+2QJsh
# IUDQtxMkzdwdeDrknq3lNHGS1yZr5Dhzq6YBT70/O3itTK37xJV77QpfMzmHQXh6
# OOmc4d0j/R0o08f56PGYX/sr2H7yRp11LB4nLCbbbxV7HhmLNriT1ObyF5lZynDw
# N7+YAN8gFk8n+2BnFqFmut1VwDophrCYoCvtlUG3OtUVmDG0YgkPCr2B2RP+v6TR
# 81fZvAT6gt4y3wSJ8ADNXcL50CN/AAvkdgIm2fBldkKmKYcJRyvmfxqkhQ/8mJb2
# VVQrH4D6wPIOK+XW+6kvRBVK5xMOHds3OBqhK/bt1nz8MIIFjTCCBHWgAwIBAgIQ
# DpsYjvnQLefv21DiCEAYWjANBgkqhkiG9w0BAQwFADBlMQswCQYDVQQGEwJVUzEV
# MBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29t
# MSQwIgYDVQQDExtEaWdpQ2VydCBBc3N1cmVkIElEIFJvb3QgQ0EwHhcNMjIwODAx
# MDAwMDAwWhcNMzExMTA5MjM1OTU5WjBiMQswCQYDVQQGEwJVUzEVMBMGA1UEChMM
# RGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQD
# ExhEaWdpQ2VydCBUcnVzdGVkIFJvb3QgRzQwggIiMA0GCSqGSIb3DQEBAQUAA4IC
# DwAwggIKAoICAQC/5pBzaN675F1KPDAiMGkz7MKnJS7JIT3yithZwuEppz1Yq3aa
# za57G4QNxDAf8xukOBbrVsaXbR2rsnnyyhHS5F/WBTxSD1Ifxp4VpX6+n6lXFllV
# cq9ok3DCsrp1mWpzMpTREEQQLt+C8weE5nQ7bXHiLQwb7iDVySAdYyktzuxeTsiT
# +CFhmzTrBcZe7FsavOvJz82sNEBfsXpm7nfISKhmV1efVFiODCu3T6cw2Vbuyntd
# 463JT17lNecxy9qTXtyOj4DatpGYQJB5w3jHtrHEtWoYOAMQjdjUN6QuBX2I9YI+
# EJFwq1WCQTLX2wRzKm6RAXwhTNS8rhsDdV14Ztk6MUSaM0C/CNdaSaTC5qmgZ92k
# J7yhTzm1EVgX9yRcRo9k98FpiHaYdj1ZXUJ2h4mXaXpI8OCiEhtmmnTK3kse5w5j
# rubU75KSOp493ADkRSWJtppEGSt+wJS00mFt6zPZxd9LBADMfRyVw4/3IbKyEbe7
# f/LVjHAsQWCqsWMYRJUadmJ+9oCw++hkpjPRiQfhvbfmQ6QYuKZ3AeEPlAwhHbJU
# KSWJbOUOUlFHdL4mrLZBdd56rF+NP8m800ERElvlEFDrMcXKchYiCd98THU/Y+wh
# X8QgUWtvsauGi0/C1kVfnSD8oR7FwI+isX4KJpn15GkvmB0t9dmpsh3lGwIDAQAB
# o4IBOjCCATYwDwYDVR0TAQH/BAUwAwEB/zAdBgNVHQ4EFgQU7NfjgtJxXWRM3y5n
# P+e6mK4cD08wHwYDVR0jBBgwFoAUReuir/SSy4IxLVGLp6chnfNtyA8wDgYDVR0P
# AQH/BAQDAgGGMHkGCCsGAQUFBwEBBG0wazAkBggrBgEFBQcwAYYYaHR0cDovL29j
# c3AuZGlnaWNlcnQuY29tMEMGCCsGAQUFBzAChjdodHRwOi8vY2FjZXJ0cy5kaWdp
# Y2VydC5jb20vRGlnaUNlcnRBc3N1cmVkSURSb290Q0EuY3J0MEUGA1UdHwQ+MDww
# OqA4oDaGNGh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydEFzc3VyZWRJ
# RFJvb3RDQS5jcmwwEQYDVR0gBAowCDAGBgRVHSAAMA0GCSqGSIb3DQEBDAUAA4IB
# AQBwoL9DXFXnOF+go3QbPbYW1/e/Vwe9mqyhhyzshV6pGrsi+IcaaVQi7aSId229
# GhT0E0p6Ly23OO/0/4C5+KH38nLeJLxSA8hO0Cre+i1Wz/n096wwepqLsl7Uz9FD
# RJtDIeuWcqFItJnLnU+nBgMTdydE1Od/6Fmo8L8vC6bp8jQ87PcDx4eo0kxAGTVG
# amlUsLihVo7spNU96LHc/RzY9HdaXFSMb++hUD38dglohJ9vytsgjTVgHAIDyyCw
# rFigDkBjxZgiwbJZ9VVrzyerbHbObyMt9H5xaiNrIv8SuFQtJ37YOtnwtoeW/VvR
# XKwYw02fc7cBqZ9Xql4o4rmUMYIDdjCCA3ICAQEwdzBjMQswCQYDVQQGEwJVUzEX
# MBUGA1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0IFRydXN0
# ZWQgRzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0YW1waW5nIENBAhALrma8Wrp/lYfG
# +ekE4zMEMA0GCWCGSAFlAwQCAQUAoIHRMBoGCSqGSIb3DQEJAzENBgsqhkiG9w0B
# CRABBDAcBgkqhkiG9w0BCQUxDxcNMjUwMjExMTQyMDE5WjArBgsqhkiG9w0BCRAC
# DDEcMBowGDAWBBTb04XuYtvSPnvk9nFIUIck1YZbRTAvBgkqhkiG9w0BCQQxIgQg
# hGfYbr9pAbDWoRIpiQIrjDnGAnnsbcbcW2/Em3xbHTAwNwYLKoZIhvcNAQkQAi8x
# KDAmMCQwIgQgdnafqPJjLx9DCzojMK7WVnX+13PbBdZluQWTmEOPmtswDQYJKoZI
# hvcNAQEBBQAEggIArzJ4Fr6wmoKcd+aYqn5ry9i9I6NXiWJRvaceTOQJ//IjwKxO
# 1eWCu9utbshwxviidJExAWgc1Qs9ieDlXS8iXHLHoTtaY9jiPB/fOgCswQO+xyCo
# 059GlsLWZa2N/sFkJ9UEx2tJJgmxrsQXXOpBtUK3R7Si8RPp55oSbHtkO++c8fQX
# yF+8UJABPmDTYu0RPx8V7vV1NK9DvyV9O7Hv7NQfdASp97FUfO4ZBZJ+pjgBP/G/
# zpd1UwZigBR9522/dpIXMSjiu8HlXgpNQZTBiTKXCCgWwrXAabEbQ39Dwre6qRuv
# 9E7VIDcHaDlVUjIaTSBto1iayTTEyqzvIlSrmj53Nx6lgLeNDZG41op2Dsj/n39C
# 7QialUIUgDa87n0zC/jap6CUGNaJG1sKbNLQWuFWrhWd+UzZTbATpbVfQczkMV6z
# UNQUrfAfdyc7ZN/k1fgyp3ISeuz6z13ahW4P/pzJstB8CtsnCTDGhK7Xf4JQw/wI
# o8skYHl1JNhCHxp/O4SGa1jIop8SUufFEotikNGDM1zHhrH5rkdRe/Zqy+IsUYP9
# 0EFOZME7ptn4R6ho0HBZNNpOC5anQXd3US7Q746FbxwgaOwbopCqHDkL3sY05n4g
# W00/cRqPgYiyLDJjM7C1WkBOhUYJME1A2L0syLpwK2g56meIrToaWHJolDQ=
# SIG # End signature block
