/*************************************************************************************************
// INTEL CONFIDENTIAL Copyright 2011-2022 Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to the source code
// ("Material") are owned by Intel Corporation or its suppliers or licensors. Title to the Material
// remains with Intel Corporation or its suppliers and licensors. The Material contains trade secrets
// and proprietary and confidential information of Intel or its suppliers and licensors. The Material is
// protected by worldwide copyright and trade secret laws and treaty provisions. No part of the
// Material may be used, copied, reproduced, modified, published, uploaded, posted, transmitted,
// distributed, or disclosed in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual property right is
// granted to or conferred upon you by disclosure or delivery of the Materials, either expressly, by
// implication, inducement, estoppel or otherwise. Any license under such intellectual property
// rights must be express and approved by Intel in writing.
//*************************************************************************************************/

#ifndef _SENSOR_INTERFACE_H_
#define _SENSOR_INTERFACE_H_


#if defined(__cplusplus)
extern "C" {
#endif

#include "sensor_types.h"
#include <acpitabl.h>


/*!
 * \brief Sensor driver interface version
 *
 * \ver.18 Add interface GetFPSMappingTable to get mulitple fps mapping table from sensor driver.
 * \ver.9 cht2 special interface SetResolutionIndex
 * \ver.8 code refactor
 * \ver.7 remove not used GetAperture field in  SENSOR_INTERFACE and add an Reserved field
 */
#define SENSOR_INTERFACE_VERSION       18
#define SENSOR_INTERFACE_SKL_VERSION   9
#define INTERFACE_ALLOCATION_TAG      'FTNI'

/*!
 * \brief Sensor Interface Standard GUID
 *
 * \{B426401D-67B7-47EF-8E70-34AAAC721DDA}
 */
static const GUID GUID_SENSOR_INTERFACE_STANDARD =
{ 0xb426401d, 0x67b7, 0x47ef, { 0x8e, 0x70, 0x34, 0xaa, 0xac, 0x72, 0x1d, 0xda } };

/*!
 * \brief SNSR_GET_PLD function.
 * This function is called to get the platform location information of sensor device, including front or rear camera.
 * It must be called when avstream driver adding sensor device, and register the information to OS.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[out]    Pld               Mandatory.\n
 *                                  The pointer of platform location information data structure ACPI_PLD_V2_BUFFER.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
*/
typedef
NTSTATUS
(* SNSR_GET_PLD) (
    PVOID Context,
    PACPI_PLD_V2_BUFFER Pld
    );

/*!
 * \brief SNSR_GET_VERSION function.
 * This function is called to get the version information of sensor device, including sensor id, version id and so on.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[out]    Version           Mandatory.\n
 *                                  The pointer of sensor version information data structure SNSR_VERSION_INFO.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(* SNSR_GET_VERSION) (
    PVOID Context,
    SNSR_VERSION_INFO *Version
    );

/*!
 * \brief SNSR_SET_EXPOSURE function.
 * This function is called to set integration related value to sensor device, including exposure time and ananlog/digital gain.
 * AE and Manual Exposure will call this function
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[in]     exposure          Mandatory.\n
 *                                  The pointer of integration related value data structure SNSR_EXPOSURE.
 *                                  Note that in order to sync up exposure time and gain, the two factor are set together.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(* SNSR_SET_EXPOSURE) (
    PVOID Context,
    SNSR_EXPOSURE exposure
    );

/*!
 * \brief SNSR_GET_EXPOSURE function.
 * This function is called to get exposure time of sensor device.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[out]    ExposureTime      Mandatory.\n
 *                                  The pointer of exposure time ULONG.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(* SNSR_GET_EXPOSURE) (
    PVOID Context,
    SNSR_EXPOSURE *ExposureTime
    );

/*!
 * \brief SNSR_GET_GAIN function.
 * This function is called to get current Gain value of sensor device.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[out]    ExposureTime      Mandatory.\n
 *                                  The pointer of get current Gain ULONG.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(* SNSR_GET_GAIN) (
    PVOID  Context,
    ULONG *Gain
    );

/*!
 * \brief SNSR_START_VIEWFINDER function.
 * This function is called to open sensor frame streaming out.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[in]     Width             Mandatory.\n
 *                                  The width of wanted resolution.
 * \param[in]     Height            Mandatory.\n
 *                                  The height of wanted resolution.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(* SNSR_START_VIEWFINDER) (
    PVOID Context,
    ULONG Width,
    ULONG Height
    );

/*!
 * \brief SNSR_STOP_VIEWFINDER function.
 * This function is called to close sensor frame streaming out.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(* SNSR_STOP_VIEWFINDER) (
    PVOID Context
    );

/*!
 * \brief SNSR_SET_FOCUS_POS function.
 * This function is called to set focus position value of vcm motor in sensor device.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[in]     FocusPos          Mandatory.\n
 *                                  The pointer of focus position in USHORT.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(* SNSR_SET_FOCUS_POS) (
    PVOID Context,
    USHORT FocusPos
    );

/*!
 * \brief SNSR_GET_FOCUS_POS function.
 * This function is called to get focus position value of vcm motor in sensor device.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[out]    FocusPos          Mandatory.\n
 *                                  The pointer of focus position in USHORT.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(* SNSR_GET_FOCUS_POS) (
    PVOID   Context,
    USHORT *FocusPos
    );

/*!
 * \brief SNSR_GET_MIPI_CONFIGS function.
 * This function is called to get sensor information that ISP needed, include mipi port index, number of mipi lanes,
 * and frame output format.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[out]    Port              Mandatory.\n
 *                                  The pointer of ISP CSI receiver mipi port, in ULONG.
 *                                  0 - MIPI_CSI2_PORT_0
 *                                  1 - MIPI_CSI2_PORT_1
 *                                  2 - MIPI_CSI2_PORT_2
 * \param[out]    LaneNum           Mandatory.\n
 *                                  The pointer of mipi lanes used for both sensor and ISP CSI receiver, in ULONG.
 *                                  There are 1, 2, 3, 4 avaiable value
 * \param[out]    OutputFormat      Mandatory.\n
 *                                  The pointer of output format, in ULONG.
 *                                  0 - BAYER_ORDER_GRBG
 *                                  1 - BAYER_ORDER_RGGB
 *                                  2 - BAYER_ORDER_BGGR
 *                                  3 - BAYER_ORDER_GBRG
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
void
(* SNSR_GET_MIPI_CONFIGS) (
    PVOID Context,
    ULONG *Port,
    ULONG *LaneNum,
    ULONG *OutputFormat
    );

/*!
 * \brief SNSR_GET_CAPS function.
 * This function is called to get sensor capability. It is called after sensor added into avstream device node
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[out]    Caps              Mandatory.\n
 *                                  The pointer sensor capability, in SNSR_CAPS.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(* SNSR_GET_CAPS) (
    PVOID Context,
    SNSR_CAPS **Caps
    );

/*!
 *\brief SNSR_GET_CAPS_EXT function
 * This function is an extension function of SNSR_GET_CAPS. It is a compatible-forward function and allows
 * sensor driver developer to add new sensor caps independently (i.e. new sensor driver can be compitable on
 * earlier ISP driver).
*/
typedef
NTSTATUS
(*SNSR_GET_CAPS_EXT) (
    PVOID Context,
    SNSR_CAPS_EXT_ID CapId,
    SNSR_CAPS_EXT_TYPE Type,
    SNSR_CAPS_EXT_DATA* Data
    );

/*!
* \brief SNSR_GET_PDAF_CAPS function.
* This function is called to get sensor PDAF capability. It is called after set sensor resolution
*
* \param[in]     Context           Mandatory.\n
*                                  The context pointer of the using sensor device.
* \param[out]    Caps              Mandatory.\n
*                                  The pointer sensor PDAF capability, in PDAF_CAPS.
* return                           Kernal mode status code, NTSTATUS.
*                                  STATUS_SUCCESS if normal, or corresponding status code
*/
typedef
NTSTATUS
(*SNSR_GET_PDAF_CAPS) (
PVOID Context,
PDAF_CAPS **Caps
);

/*!
 * \brief SNSR_GET_RESOLUTIONS function.
 * This function is called to get sensor supported resolutions. It is called after sensor added into avstream device node
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[out]    StillRes          Mandatory.\n
 *                                  The pointer still pin resolutions supported by sensor, in SNSR_RESOLUTION_SET.
 * \param[out]    VideoRes          Mandatory.\n
 *                                  The pointer video pin resolutions supported by sensor, in SNSR_RESOLUTION_SET.
 * \param[out]    PreviewRes        Mandatory.\n
 *                                  The pointer preview pin resolutions supported by sensor, in SNSR_RESOLUTION_SET.
 * \param[out]    RawRes            Mandatory.\n
 *                                  The pointer raw pin resolutions supported by sensor, in SNSR_RESOLUTION_SET.
 *                                  It is used for engineer debug purpose
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(* SNSR_GET_RESOLUTIONS) (
    PVOID Context,
    SNSR_RESOLUTION_SET **StillRes,
    SNSR_RESOLUTION_SET **VideoRes,
    SNSR_RESOLUTION_SET **PreviewRes,
    SNSR_RESOLUTION_SET **RawRes
    );

typedef
NTSTATUS
(*SNSR_GET_PDAF_RESOLUTIONS) (
    PVOID Context,
    PDAF_RESOLUTION_SET **PDAFRes
);

/*!
 * \brief SNSR_SET_RESOLUTIONS function.
 * This function is called to set a certain sensor resolution before streaming on.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[in]     mode              Mandatory.\n
 *                                  The sensor mode, in SNSR_MODE enum type.
 * \param[in]     Width             Mandatory.\n
 *                                  The width of wanted resolution.
 * \param[in]     Height            Mandatory.\n
 *                                  The height of wanted resolution.
 * \param[in]     frameRate         Mandatory.\n
 *                                  The frame rate of wanted setting, in frame per second.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(* SNSR_SET_RESOLUTIONS) (
    PVOID Context,
    SNSR_MODE mode,
    ULONG Width,
    ULONG Height,
    LONG frameRate
    );

/*!
 * \brief SNSR_GET_FPS_MAPPING_TABLE function.
 * This function is called to get sensor fps mapping table
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[out]    Status            Mandatory.\n
 *                                  The pointer of focus status, in FPS_MAPPING_TABLE.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(*SNSR_GET_FPS_MAPPING_TABLE) (
    PVOID Context,
    FPS_MAPPING_TABLE * FPSMappingTable
    );

/*!
 * \brief SNSR_GET_FOCUS_STATUS function.
 * This function is called to get focus status.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[out]    Status            Mandatory.\n
 *                                  The pointer of focus status, in PULONG.
 *                                  0 - stopped
 *                                  1 - moving
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(* SNSR_GET_FOCUS_STATUS) (
    PVOID Context,
    PULONG Status
    );

/*!
 * \brief SNSR_GET_FOCUS_HP_STATUS function.
 * This function is called to get focus home postion status.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[out]    Status            Mandatory.\n
 *                                  The pointer of focus home postion status, in PULONG.
 *                                  0 - incomplete
 *                                  1 - complete
 *                                  2 - error
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(* SNSR_GET_FOCUS_HP_STATUS) (
    PVOID Context,
    PULONG Status
    );

/*!
 * \brief SNSR_GET_DATA function.
 * This function is called to get calibration and mode data.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[in, out]    Size          Mandatory.\n
 *                                  [in] pointer of the size of buffer to hold the data, in PULONG.
 *                                  [out] pointer of the size of returned data, in PULONG.
 * \param[out]    Buffer            Mandatory.\n
 *                                  The pointer of the buffer to hold the data, in PUINT8.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(* SNSR_GET_DATA) (
    PVOID Context,
    PULONG Size,
    PUINT8 Buffer
    );

/*!
 * \brief SNSR_SIGNAL_VBLANK function.
 * This function is called for Signal VBlank event to the sensor driver.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(* SNSR_SIGNAL_VBLANK) (
    PVOID  Context
    );


/*!
 * \brief SNSR_SET_IDLE function.
 * This function is called set sensor device power into D3 mode.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(* SNSR_SET_IDLE) (
    PVOID Context
    );

/*!
 * \brief SNSR_SET_ACTIVE function.
 * This function is called set sensor device power into D0 mode.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(* SNSR_SET_ACTIVE) (
    PVOID Context
    );


/*!
 * \brief SNSR_EEPROM Get/Set function.
 * This function is called get/set data from/to sensor eeprom device.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[in]     Addr              Mandatory.\n
 *                                  EEPROM address wanted to be accessed.
 * \param[out/in] pData             Mandatory.\n
 *                                  EEPROM data buffer.
 * \param[in]     Length            Mandatory.\n
 *                                  EEPROM data length.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(* SNSR_EEPROM) (
    PVOID Context,
    UINT16 Addr,
    UINT8* pData,
    INT32 Length
    );

/*!
 * \brief SNSR_SEND_IMAGE_TO_FIFO function.
 * This function is called to send image to FIFO. It is only used for fifo sensor mode.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[in]     Stream            Mandatory.\n
 *                                  Image data buffer point.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(* SNSR_SEND_IMAGE_TO_FIFO) (
    PVOID Context,
    PVOID Stream
    );


/*!
 * \brief SNSR_SET_SEND_INPUT_IMAGE_INTF function.
 * This function is called to set the information of image to FIFO. It is only used for fifo sensor mode.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[in]     Stream            Mandatory.\n
 *                                  Information point Image data for FIFO point, in SEND_INPUT_IMAGE.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
void
(* SNSR_SET_SEND_INPUT_IMAGE_INTF) (
    PVOID Context,
    SEND_INPUT_IMAGE SendProc
    );

/*!
 * \brief SNSR_GET_ULONG (Binning, Framerate) function.
 * This function is called to get the Binning, Framerate information of sensor.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[out]     pData            Mandatory.\n
 *                                  Pointer of value that wanted (Binning, Framerate).
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(* SNSR_GET_ULONG) (
    PVOID Context,
    PULONG pData
    );

/*!
 * \brief SNSR_GET_INFO  function.
 * This function is called to get information of sensor.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[out]    pInfo             Mandatory.\n
 *                                  Pointer of sensor information, in PSNSR_INFO.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(* SNSR_GET_INFO) (
    PVOID Context,
    PVOID pInfo
    );

/*!
 * \brief SNSR_SET_VCM_CONFIGURATION  function.
 * This function is called to set vcm init data of sensor vcm mdoe.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[in]     pSnsrVcmData      Mandatory.\n
 *                                  Pointer of sensor vcm init data, in SNSR_VCMDATA.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(* SNSR_SET_VCM_CONFIGURATION) (
    PVOID Context,
    SNSR_VCMDATA* pSnsrVcmData
    );

/*!
 * \brief SNSR_SET_EXT_FEATURE  function.
 * This function is called to set sensor feature related data.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[in]     feature           Mandatory.\n
 *                                  Sensor feature ID wanted, in SNSR_EXT_FEATURE_ID.
 * \param[in]     Value             Mandatory.\n
 *                                  Pointer of data, in ULONG_PTR.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(* SNSR_SET_EXT_FEATURE) (
    PVOID Context,
    SNSR_EXT_FEATURE_ID feature,
    ULONG_PTR Value
    );

/*!
 * \brief SNSR_GET_EXT_FEATURE  function.
 * This function is called to get sensor feature related data.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[in]     feature           Mandatory.\n
 *                                  Sensor feature ID wanted, in SNSR_EXT_FEATURE_ID.
 * \param[out]    pValue            Mandatory.\n
 *                                  Pointer of data, in ULONG_PTR.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(* SNSR_GET_EXT_FEATURE) (
    PVOID Context,
    SNSR_EXT_FEATURE_ID feature,
    PULONG pValue
    );


/*!
* \brief SNSR_SET_RESOLUTIONS_INDEX function.
* This function is called to set a certain sensor resolution before streaming on.
*
* \param[in]     Context           Mandatory.\n
*                                  The context pointer of the using sensor device.
* \param[in]     mode              Mandatory.\n
*                                  The sensor mode, in SNSR_MODE enum type.
* \param[in]     index             Mandatory. \n
*                                  The index of the resolution to use
* return                           Kernal mode status code, NTSTATUS.
*                                  STATUS_SUCCESS if normal, or corresponding status code
*/
typedef
NTSTATUS
(*SNSR_SET_RESOLUTIONS_INDEX) (
    PVOID Context,
    SNSR_MODE mode,
    ULONG index
    );

//*********************************************************************
//
// Routine Description:
//     GET CONTROLLOGIC INFO
//
// Arguments:
//      info - CONTROLLOGIC_INFO
// Returns Value:
//     STATUS_SUCCESS if normal, or other corresponding status code
//     if unexpected event happens.
//
//*********************************************************************
typedef
NTSTATUS
(*SNSR_GET_CONTROLLOGIC_INFO) (
    PVOID Context,
    PVOID info
    );

/*!
 * \brief SNSR_SET_REGISTER function.
 * This function is called set register to sensor device.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[in]     Addr              Mandatory.\n
 *                                  Sensor register address wanted to be set.
 * \param[in]     Data              Mandatory.\n
 *                                  Sensor register data wanted to be set.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(*SNSR_SET_REGISTER) (
    PVOID Context,
    UINT16 Addr,
    UINT16 Data
    );
/*!
 * \brief SNSR_GET_REGISTER function.
 * This function is called set register to sensor device.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[in]     Addr              Mandatory.\n
 *                                  Sensor register address wanted to be get.
 * \param[out]    pData             Mandatory.\n
 *                                  Sensor register data wanted to be get.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(*SNSR_GET_REGISTER) (
    PVOID Context,
    UINT16 Addr,
    UINT16* pData
    );

/*!
 * \brief SNSR_SET_REGISTER_BULK function.
 * This function is called set register bulk to sensor device.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[in]     pInputReg         Mandatory.\n
 *                                  Sensor register address/width/data to set.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(*SNSR_SET_REGISTER_BULK) (
    PVOID Context,
    PSNSR_REG_LIST pInputReg
    );
/*!
 * \brief SNSR_GET_REGISTER_BULK function.
 * This function is called set register to sensor device.
 *
 * \param[in]     Context           Mandatory.\n
 *                                  The context pointer of the using sensor device.
 * \param[in]     pInputReg         Mandatory.\n
 *                                  Sensor register address and width to get register data.
 * \param[out]    pOutputReg        Mandatory.\n
 *                                  Sensor register data.
 * return                           Kernal mode status code, NTSTATUS.
 *                                  STATUS_SUCCESS if normal, or corresponding status code
 */
typedef
NTSTATUS
(*SNSR_GET_REGISTER_BULK) (
    PVOID Context,
    PSNSR_REG_LIST pInputReg,
    PSNSR_REG_LIST pOutputReg
    );

typedef
NTSTATUS
(*SNSR_SET_CALIBRATION_DATA) (
    PVOID Context,
    PUINT8 Buffer,
    ULONG Size
    );

typedef
NTSTATUS
(*SNSR_SET_PLATFORM_INFO) (
    PVOID Context,
    PPLAT_INFO PlatformInfo
);

/*!
* \brief SNSR_SET_DUAL_EXPOSURE function.
* This function is called to set integration related value to sensor device, including exposure time and ananlog/digital gain.
* AE and Manual Exposure will call this function
*
* \param[in]     Context           Mandatory.\n
*                                  The context pointer of the using sensor device.
* \param[in]     Exposure          Mandatory.\n
*                                  The pointer of multiple integrations related value data structure SNSR_EXPOSURE.
*                                  Note that in order to sync up exposure time and gain, the two factor are set together.
* \param[in]     Count             Mandatory.\n
*                                  The number of multiple integrations.
* return                           kernel mode status code, NTSTATUS.
*                                  STATUS_SUCCESS if normal, or corresponding status code
*/
typedef
NTSTATUS
(*SNSR_SET_MULTIPLE_EXPOSURE) (
    PVOID Context,
    SNSR_EXPOSURE *Exposure,
    USHORT Count
);

/*!
 * \brief sensor driver interface function structre.
 */
typedef struct _SENSOR_INTERFACE
{
    struct _INTERFACE              InterfaceHeader; // kcom.h also defines an INTERFACE structure, we want the one from wdm.h here.
    SNSR_GET_VERSION               GetVersion;          /*!< Mandatory. Function to get sensor version information. */
    SNSR_GET_PLD                   GetPld;              /*!< Mandatory. Function to get platform location information. */
    SNSR_GET_MIPI_CONFIGS          GetMIPIConfigs;      /*!< Mandatory. Function to get mipi related configuration. */
    SNSR_GET_CAPS                  GetCaps;             /*!< Mandatory. Function to get sensor capability. */
    SNSR_GET_RESOLUTIONS           GetResolutions;      /*!< Mandatory. Function to get sensor supported resolutions. */
    SNSR_GET_PDAF_RESOLUTIONS      GetPDAFResolutions;  /*!< Optinal. Function to get sensor supported PDAF resolutions. */
    SNSR_SET_RESOLUTIONS           SetResolution;       /*!< Mandatory. Function to set certain sensor resolution. */
    SNSR_SET_EXPOSURE              SetExposure;         /*!< Mandatory. Function to set exposure time. */
    SNSR_GET_EXPOSURE              GetExposure;         /*!< Mandatory. Function to get exposure time. */
    SNSR_GET_GAIN                  GetGain;             /*!< Mandatory. Function to get gain. */
    SNSR_START_VIEWFINDER          StartViewfinder;     /*!< Mandatory. Function to open sensor stream. */
    SNSR_STOP_VIEWFINDER           StopViewfinder;      /*!< Mandatory. Function to close sensor stream. */
    SNSR_SET_FOCUS_POS             SetFocusPos;         /*!< Mandatory. Function to set vcm focus position. */
    SNSR_GET_FOCUS_POS             GetFocusPos;         /*!< Mandatory. Function to get vcm focus position. */
    SNSR_GET_FOCUS_STATUS          GetFocusStatus;      /*!< Mandatory. Function to get vcm focus status. */
    SNSR_GET_FOCUS_HP_STATUS       GetFocusHPStatus;    /*!< Mandatory. Function to get vcm focus home position status. */
    SNSR_GET_DATA                  GetCalibrationData;  /*!< Mandatory. Function to get sensor calibration data. */
    SNSR_GET_DATA                  GetModeData;         /*!< Mandatory. Function to get sensor mode data. */
    SNSR_SET_ACTIVE                SetActive;           /*!< Mandatory. Function to set sensor device into D0 power mode. */
    SNSR_SET_IDLE                  SetIdle;             /*!< Mandatory. Function to set sensor device into D3 power mode. */
    SNSR_SEND_IMAGE_TO_FIFO        SendImageToFIFO;     /*!< Mandatory. Function to send image to fifo mode. */
    SNSR_SET_SEND_INPUT_IMAGE_INTF SetSendInputImageIntf;/*!< Mandatory. Function to send image information to fifo mode. */
    SNSR_EEPROM                    SetEEPRom;           /*!< Optional. Function to set eeprom. */
    SNSR_EEPROM                    GetEEPRom;           /*!< Mandatory. Function to get eeprom data. */
    SNSR_GET_ULONG                 GetBinningMode;      /*!< Mandatory. Function to get sensor binning mode. */
    SNSR_GET_ULONG                 GetFrameRate;        /*!< Mandatory. Function to get sensor frame rate. */
    SNSR_GET_INFO                  GetSensorInfo;       /*!< Mandatory. Function to get sensor information. */
    SNSR_SIGNAL_VBLANK             SignalVBlank;        /*!< Optional. Function for sensor v-blank. */
    SNSR_SET_VCM_CONFIGURATION     SetVcmConfiguration; /*!< Mandatory. Function to set vcm init configuration. */
    SNSR_SET_EXT_FEATURE           SetExtFeature;       /*!< Mandatory. Function to set sensor feature data. */
    SNSR_GET_EXT_FEATURE           GetExtFeature;       /*!< Mandatory. Function to get sensor feature data. */
    PVOID                          Reserved;            /*!< Optional. Researved. */
    SNSR_SET_REGISTER              SetRegister;
    SNSR_GET_REGISTER              GetRegister;
    SNSR_SET_REGISTER_BULK         SetRegisterBulk;
    SNSR_GET_REGISTER_BULK         GetRegisterBulk;
    SNSR_SET_RESOLUTIONS_INDEX     SetResolutionIndex;  /*!< Optional. Functuion to set certain sensor resolution givin particular extra parameters. */  
    SNSR_GET_PDAF_CAPS             GetPDAFCaps;         /*!< Optional. Function to get PDAF capability. */
    SNSR_SET_CALIBRATION_DATA      SetSensorCalibrationData; /*< Optional. Function to set Sensor related calibration data. */
    SNSR_GET_CONTROLLOGIC_INFO     GetControlLogicInfo;
    SNSR_SET_PLATFORM_INFO         SetPlatformInfo;
    SNSR_SET_MULTIPLE_EXPOSURE     SetMultipleExposure;
    SNSR_GET_FPS_MAPPING_TABLE     GetFPSMappingTable;  /*!< Optional.  Function to get fps mapping table. */
} SENSOR_INTERFACE;


#if defined(__cplusplus)
}
#endif
#endif //_SENSOR_INTERFACE_H_
